#include "carrier_dumper.h"

#include <travel/rasp/rasp_data/dumper/lib/fetcher/fake_fetcher.h>
#include <travel/rasp/rasp_data/dumper/lib/object_writer/fake_object_writer.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/fabrics.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/json.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

class TCarrierDumperTest: public TTestBase {
private:
    UNIT_TEST_SUITE(TCarrierDumperTest);
    UNIT_TEST(TestEmpty);
    UNIT_TEST(TestOneRecord);
    UNIT_TEST(TestUnknownHomeStation);
    UNIT_TEST(TestEmptyHomeStation);
    UNIT_TEST(TestUnknownTransportType);
    UNIT_TEST(TestUnknownCountry);
    UNIT_TEST_SUITE_END();

private:
    TFakeFetcher Fetcher_;
    TFakeObjectWriter<TCarrier> ObjectWriter_;
    THolder<TCarrierDumper> Dumper_;

    const i32 DefaultHomeStationId_ = 213;
    const i32 DefaultId_ = 999;
    const ui32 DefaultCountryId_ = 100;

    TSet<i32> StationIds_;
    TSet<ui32> CountryIds_;

public:
    void SetUp() override {
        StationIds_ = {DefaultHomeStationId_};
        CountryIds_ = {DefaultCountryId_};

        Fetcher_ = {};
        ObjectWriter_ = TFakeObjectWriter<TCarrier>();
        Dumper_ = MakeHolder<TCarrierDumper>(Fetcher_, ObjectWriter_);
    }

    TVector<TString> CreateRow() {
        return {
            ToString(DefaultId_),              // id
            "some address",                    // address
            ToString(DefaultHomeStationId_),   // home_station_id
            "iata_value",                      // iata
            "icao_value",                      // icao
            "title_value",                     // title
            "title_en_value",                  // title_en
            "url_value",                       // url
            "contact_info_value",              // contact_info
            "description_value",               // description
            "logo_value",                      // logo
            ToString(DefaultCountryId_),       // country_id
            "2",                               // transport_type
            "0",                               // hidden
            "0",                               // strange
            "email_value",                     // email
            "sirena_value",                    // sirena_id
            "phone_value",                     // phone
            "phone_booking_value",             // phone_booking
            "meta_title_value",                // meta_title
            "meta_description_value",          // meta_description_value
            "supplier_code_value",             // supplier_code
            "title_ru_value",                  // title_ru
            "title_uk_value",                  // title_uk
            "title_tr_value",                  // title_tr
            "short_title_value",               // short_title
            "short_title_ru_value",            // short_title_ru
            "short_title_en_value",            // short_title_en
            "short_title_uk_value",            // short_title_uk
            "short_title_tr_value",            // short_title_tr
            "0",                               // is_freight
            "icon_value",                      // icon
            "logo_mono_value",                 // logo_mono
            "logo_bg_color_value",             // logo_bg_color
            "3",                               // priority
            "icao_ru_value",                   // icao_ru
            "svg_logo_value",                  // svg_logo
            "9999",                            // alliance_id
            "bonus_name_value",                // bonus_name
            "bonus_name_ru_value",             // bonus_name_ru
            "bonus_name_en_value",             // bonus_name_en
            "bonus_name_tr_value",             // bonus_name_tr
            "bonus_name_uk_value",             // bonus_name_uk
            "svg_to_png_logo_value",           // svg_to_png_logo
            "registration_phone_value",        // registration_phone
            "registration_phone_ru_value",     // registration_phone_ru
            "registration_phone_en_value",     // registration_phone_en
            "registration_phone_tr_value",     // registration_phone_tr
            "registration_phone_uk_value",     // registration_phone_uk
            "registration_url_value",          // registration_url
            "registration_url_ru_value",       // registration_url_ru
            "registration_url_en_value",       // registration_url_en
            "registration_url_tr_value",       // registration_url_tr
            "registration_url_uk_value",       // registration_url_uk
            "tis_code_value",                  // tis_code
            "express_code_value",              // express_code
            "yandex_avia_code_value",          // yandex_avia_code
        };
    }
    TCarrier CreateModel() {
        TCarrier record;
        record.SetId(DefaultId_);
        record.SetAddress("some address");
        record.SetHomeStationId(DefaultHomeStationId_);
        record.SetIata("iata_value");
        record.SetIcao("icao_value");
        record.SetTitle("title_value");
        record.SetTitleEn("title_en_value");
        record.SetUrl("url_value");
        record.SetContactInfo("contact_info_value");
        record.SetDescription("description_value");
        record.SetLogo("logo_value");
        record.SetCountryId(DefaultCountryId_);
        record.SetTransportType(TTransport::TYPE_PLANE);
        record.SetEmail("email_value");
        record.SetSirenaId("sirena_value");
        record.SetPhone("phone_value");
        record.SetPhoneBooking("phone_booking_value");
        record.SetMetaTitle("meta_title_value");
        record.SetMetaDescription("meta_description_value");
        record.SetSupplierCode("supplier_code_value");
        record.SetTitleRu("title_ru_value");
        record.SetTitleUk("title_uk_value");
        record.SetTitleTr("title_tr_value");
        record.SetShortTitle("short_title_value");
        record.SetShortTitleRu("short_title_ru_value");
        record.SetShortTitleEn("short_title_en_value");
        record.SetShortTitleUk("short_title_uk_value");
        record.SetShortTitleTr("short_title_tr_value");
        record.SetIcon("icon_value");
        record.SetLogo("logo_value");
        record.SetLogoMono("logo_mono_value");
        record.SetLogoBgColor("logo_bg_color_value");
        record.SetPriority(3);
        record.SetIcaoRu("icao_ru_value");
        record.SetSvgLogo("svg_logo_value");
        record.SetAllianceId(9999);
        record.SetBonusName("bonus_name_value");
        record.SetBonusNameRu("bonus_name_ru_value");
        record.SetBonusNameEn("bonus_name_en_value");
        record.SetBonusNameTr("bonus_name_tr_value");
        record.SetBonusNameUk("bonus_name_uk_value");
        record.SetSvg2PngLogo("svg_to_png_logo_value");
        record.SetRegistrationPhone("registration_phone_value");
        record.SetRegistrationPhoneRu("registration_phone_ru_value");
        record.SetRegistrationPhoneEn("registration_phone_en_value");
        record.SetRegistrationPhoneTr("registration_phone_tr_value");
        record.SetRegistrationPhoneUk("registration_phone_uk_value");
        record.SetRegistrationUrl("registration_url_value");
        record.SetRegistrationUrlRu("registration_url_ru_value");
        record.SetRegistrationUrlEn("registration_url_en_value");
        record.SetRegistrationUrlTr("registration_url_tr_value");
        record.SetRegistrationUrlUk("registration_url_uk_value");
        record.SetTisCode("tis_code_value");
        record.SetExpressCode("express_code_value");
        record.SetYandexAviaCode("yandex_avia_code_value");
        return record;
    }
    void TestEmpty();
    void TestOneRecord();
    void TestUnknownHomeStation();
    void TestEmptyHomeStation();
    void TestUnknownTransportType();
    void TestUnknownCountry();
};

UNIT_TEST_SUITE_REGISTRATION(TCarrierDumperTest);

void TCarrierDumperTest::TestEmpty() {
    Fetcher_.Add({});
    UNIT_ASSERT_EXCEPTION(Dumper_->Dump(StationIds_, CountryIds_), yexception);
}

void TCarrierDumperTest::TestOneRecord() {
    Fetcher_.Add({CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(StationIds_, CountryIds_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TCarrier>{CreateModel()}));
}

void TCarrierDumperTest::TestUnknownHomeStation() {
    auto r = CreateRow();
    r[Dumper_->Query.HomeStationIdField] = "666";
    Fetcher_.Add({r, CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(StationIds_, CountryIds_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TCarrier>{CreateModel()}));
}

void TCarrierDumperTest::TestEmptyHomeStation() {
    auto r = CreateRow();
    r[Dumper_->Query.HomeStationIdField] = "";
    Fetcher_.Add({r});

    auto m = CreateModel();
    m.SetHomeStationId(0);

    UNIT_ASSERT_EQUAL(Dumper_->Dump(StationIds_, CountryIds_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TCarrier>{m}));
}

void TCarrierDumperTest::TestUnknownTransportType() {
    auto r = CreateRow();
    r[Dumper_->Query.TransportTypeField] = "666";
    Fetcher_.Add({r, CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(StationIds_, CountryIds_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TCarrier>{CreateModel()}));
}

void TCarrierDumperTest::TestUnknownCountry() {
    auto r = CreateRow();
    r[Dumper_->Query.CountryIdField] = "666";
    Fetcher_.Add({r, CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(StationIds_, CountryIds_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TCarrier>{CreateModel()}));
}
