#include "country_dumper.h"

using namespace NRasp::NData;

namespace NRasp {
    namespace NDumper {
        TCountrySelectQuery::TCountrySelectQuery()
            : TSelectQuery("www_country", false)
            , IdField(AddField("id"))
            , GeoIdField(AddField("_geo_id"))
            , CodeField(AddField("code"))
            , TitleField(AddField("title"))
            , Code3Field(AddField("code3"))
            , KladrIdField(AddField("_kladr_id"))
            , DomainZoneField(AddField("domain_zone"))
            , CurrencyIdField(AddField("currency_id"))
            , LanguageField(AddField("language"))
            , ModifiedAtField(AddField("modified_at"))

            , TitleRuField(AddField("title_ru"))
            , TitleRuGenitiveField(AddField("title_ru_genitive"))
            , TitleRuLocativeField(AddField("title_ru_locative"))
            , TitleRuLocativePrepositionField(AddField("title_ru_preposition_v_vo_na"))
            , TitleEnField(AddField("title_en"))
            , TitleUkField(AddField("title_uk"))
            , TitleUkAccusativeField(AddField("title_uk_accusative"))
            , TitleTrField(AddField("title_tr"))
        {
        }

        TCountryDumper::TCountryDumper(NRasp::NDumper::IFetcher& fetcher,
                                       IObjectWriter<NData::TCountry>& writer,
                                       const ILinguisticProvider& linguisticProvider)
            : Query({})
            , Fetcher_(fetcher)
            , Writer_(writer)
            , LinguisticProvider_(linguisticProvider)
        {
        }

        TSet<ui32> TCountryDumper::Dump() {
            INFO_LOG << "Run the Country dumper" << Endl;
            Fetcher_.InitQuery(Query.Str());

            TSet<ui32> countryIds;
            TVector<TString> row(Query.Size());
            while (Fetcher_.GetNext(&row)) {
                TCountry record;
                const auto id = ParseId(row[Query.IdField]);

                record.SetId(id);
                record.SetGeoId(ParseId(row[Query.GeoIdField]));
                record.SetCode(row[Query.CodeField]);
                record.SetTitleDefault(row[Query.TitleField]);

                record.SetCode3(row[Query.Code3Field]);
                record.SetKladrId(row[Query.KladrIdField]);
                record.SetDomainZone(row[Query.DomainZoneField]);
                record.SetCurrencyId(ParseId(row[Query.CurrencyIdField]));
                record.SetLanguage(row[Query.LanguageField]);
                record.MutableModifiedAt()->set_seconds(ParseI32(row[Query.ModifiedAtField]));

                record.MutableTitle()->MutableRu()->SetNominative(row[Query.TitleRuField]);
                record.MutableTitle()->MutableRu()->SetPrepositional(row[Query.TitleRuLocativeField]);
                record.MutableTitle()->MutableRu()->SetLocativePreposition(row[Query.TitleRuLocativePrepositionField]);
                record.MutableTitle()->MutableRu()->SetGenitive(row[Query.TitleRuGenitiveField]);

                record.MutableTitle()->MutableEn()->SetNominative(row[Query.TitleEnField]);

                record.MutableTitle()->MutableTr()->SetNominative(row[Query.TitleTrField]);

                record.MutableTitle()->MutableUk()->SetNominative(row[Query.TitleUkField]);
                record.MutableTitle()->MutableUk()->SetAccusative(row[Query.TitleUkAccusativeField]);

                if (record.GetGeoId()) {
                    const auto& ruLings = LinguisticProvider_.GetLinguistics(record.GetGeoId(), "ru");
                    if (record.GetTitle().GetRu().GetNominative().empty() && !ruLings.NominativeCase.empty()) {
                        record.MutableTitle()->MutableRu()->SetNominative(TString(ruLings.NominativeCase));
                    };
                    if (record.GetTitle().GetRu().GetPrepositional().empty() && !ruLings.PrepositionalCase.empty()) {
                        record.MutableTitle()->MutableRu()->SetPrepositional(TString(ruLings.PrepositionalCase));
                    };
                    if (record.GetTitle().GetRu().GetLocativePreposition().empty() && !ruLings.Preposition.empty()) {
                        record.MutableTitle()->MutableRu()->SetLocativePreposition(TString(ruLings.Preposition));
                    };
                    if (record.GetTitle().GetRu().GetGenitive().empty() && !ruLings.GenitiveCase.empty()) {
                        record.MutableTitle()->MutableRu()->SetGenitive(TString(ruLings.GenitiveCase));
                    };

                    const auto& ukLings = LinguisticProvider_.GetLinguistics(record.GetGeoId(), "uk");
                    if (record.GetTitle().GetUk().GetNominative().empty() && !ukLings.NominativeCase.empty()) {
                        record.MutableTitle()->MutableUk()->SetNominative(TString(ukLings.NominativeCase));
                    };
                    if (record.GetTitle().GetUk().GetAccusative().empty() && !ukLings.AccusativeCase.empty()) {
                        record.MutableTitle()->MutableUk()->SetAccusative(TString(ukLings.AccusativeCase));
                    };

                    const auto& trLings = LinguisticProvider_.GetLinguistics(record.GetGeoId(), "tr");
                    if (record.GetTitle().GetTr().GetNominative().empty() && !trLings.NominativeCase.empty()) {
                        record.MutableTitle()->MutableTr()->SetNominative(TString(trLings.NominativeCase));
                    };

                    const auto& enLings = LinguisticProvider_.GetLinguistics(record.GetGeoId(), "en");
                    if (record.GetTitle().GetEn().GetNominative().empty() && !enLings.NominativeCase.empty()) {
                        record.MutableTitle()->MutableEn()->SetNominative(TString(enLings.NominativeCase));
                    };
                };

                countryIds.insert(id);
                Writer_.Write(record);
            };

            if (countryIds.empty()) {
                ythrow yexception() << "Country dumper has no records.";
            }
            INFO_LOG << "The Country dumper has dumped: total count of entries is " << countryIds.size() << Endl;

            return countryIds;
        }
    }
}
