#include "district_dumper.h"

#include <travel/rasp/rasp_data/dumper/lib/fetcher/fake_fetcher.h>
#include <travel/rasp/rasp_data/dumper/lib/object_writer/fake_object_writer.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/json.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

class TDistrictDumperTest: public TTestBase {
private:
    UNIT_TEST_SUITE(TDistrictDumperTest);
    UNIT_TEST(TestEmpty);
    UNIT_TEST(TestOneRecord);
    UNIT_TEST(TestUnknownRegion);
    UNIT_TEST(TestIsNotDisputedTerritory);
    UNIT_TEST_SUITE_END();

private:
    TFakeFetcher Fetcher_;
    TFakeObjectWriter<TDistrict> ObjectWriter_;
    THolder<TDistrictDumper> Dumper_;

    const ui32 DefaultId_ = 999;
    const i32 DefaultCapitalSettlementId_ = 213;
    const ui32 DefaultRegionId_ = 10000;
    TSet<ui32> RegionIds_;

public:
    void SetUp() override {
        Fetcher_ = {};
        ObjectWriter_ = TFakeObjectWriter<TDistrict>();
        Dumper_ = MakeHolder<TDistrictDumper>(Fetcher_, ObjectWriter_);

        RegionIds_ = {DefaultRegionId_};
    }

    TVector<TString> CreateRow() {
        return {
            ToString(DefaultId_), // id

            "title_value", // title

            ToString(DefaultCapitalSettlementId_), // settlement_id
            ToString(DefaultRegionId_),            // region_id

            "1", // disputed_territory
        };
    }

    TDistrict CreateModel() {
        TDistrict record;
        record.SetId(DefaultId_);
        record.SetTitleDefault("title_value");
        record.SetCapitalSettlementId(DefaultCapitalSettlementId_);
        record.SetRegionId(DefaultRegionId_);
        record.SetIsDisputedTerritory(true);
        return record;
    }
    void TestEmpty();
    void TestOneRecord();
    void TestUnknownRegion();
    void TestIsNotDisputedTerritory();
};

UNIT_TEST_SUITE_REGISTRATION(TDistrictDumperTest);

void TDistrictDumperTest::TestEmpty() {
    Fetcher_.Add({});

    UNIT_ASSERT_EXCEPTION(Dumper_->Dump(RegionIds_), yexception);
}

void TDistrictDumperTest::TestOneRecord() {
    Fetcher_.Add({CreateRow()});

    auto ids = Dumper_->Dump(RegionIds_);
    UNIT_ASSERT_EQUAL(ids, TSet<ui32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TDistrict>{CreateModel()}));
}

void TDistrictDumperTest::TestUnknownRegion() {
    auto row = CreateRow();
    row[Dumper_->Query.RegionIdField] = "666";
    Fetcher_.Add({row, CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(RegionIds_), TSet<ui32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TDistrict>{CreateModel()}));
}

void TDistrictDumperTest::TestIsNotDisputedTerritory() {
    auto row = CreateRow();
    row[Dumper_->Query.IsDisputedTerritoryField] = "0";
    auto model = CreateModel();
    model.SetIsDisputedTerritory(false);

    Fetcher_.Add({row});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(RegionIds_), TSet<ui32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TDistrict>{model}));
}
