#include "rasp_data_dumper.h"

using namespace NRasp::NData;

namespace NRasp {
    namespace NDumper {
        TRaspDataDumper::TRaspDataDumper(const TString& rootPath, const TString& GeobasePath_,
            const TString& tankerAuthToken, const TMySqlConfig& mySqlConfig)
            : RootPath_(rootPath)
            , GeobasePath_(GeobasePath_)
            , TankerAuthToken_(tankerAuthToken)
            , MySqlConfig_(mySqlConfig)
        {
        }

        TString TRaspDataDumper::MakeFilePath(const TString& name) {
            return RootPath_ + '/' + name + ".bin";
        }

        void TRaspDataDumper::Dump() {
            INFO_LOG << "Let's do it!" << Endl;

            INFO_LOG << "Reading the geo base data from " << GeobasePath_ << Endl;
            NGeobase::TLookup lookup(GeobasePath_);

            THolder<TTimeZoneProvider> tzProvider;
            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TTimeZone> writer(MakeFilePath("timezone"));
                TTimeZoneLoader tzLoader(fetcher, writer);
                tzProvider = MakeHolder<TTimeZoneProvider>(tzLoader.Load(), "Europe/Moscow");
            }

            {
                THttpFetcher fetcher(TankerAuthToken_);
                TProtobufWriter<TReadableTimezone> writer(MakeFilePath("readable_timezone"));
                TReadableTimezoneTankerDumper readableTimezoneTankerDumper(fetcher, writer);
                readableTimezoneTankerDumper.Dump();
            }

            TMap<TString, i32> currencyCodeToId;
            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TCurrency> writer(MakeFilePath("currency"));
                TCurrencyDumper currencyDumper(fetcher, writer);
                currencyCodeToId = currencyDumper.Dump();
            }

            TSet<i32> supplierIds;
            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TSupplier> writer(MakeFilePath("supplier"));
                TSupplierDumper supplierDumper(fetcher, writer);
                supplierIds = supplierDumper.Dump();
            }

            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TNamedTrain> writer(MakeFilePath("named_train"));
                TNamedTrainDumper namedTrainDumper(fetcher, writer);
                namedTrainDumper.Dump();
            }

            THashMap<i32, THashMap<i32, TString>> stationIdToCodes;
            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TStationCode> writer(MakeFilePath("station_code"));
                TStationCodeDumper stationCodeDumper(fetcher, writer);
                stationIdToCodes = stationCodeDumper.Dump();
            }

            TSet<ui32> countryIds;
            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TCountry> writer(MakeFilePath("country"));
                TGeobaseLinguisticProvider linguisticProvider{lookup};
                TCountryDumper countryDumper(fetcher, writer, linguisticProvider);
                countryIds = countryDumper.Dump();
            }

            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TPointSynonym> writer(MakeFilePath("pointsynonym"));
                TPointSynonymDumper pointSynonymDumper(fetcher, writer);
                pointSynonymDumper.Dump();
            }

            TSet<ui32> regionIds;
            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TRegion> writer(MakeFilePath("region"));
                TGeobaseLinguisticProvider linguisticProvider{lookup};
                TRegionDumper regionDumper(fetcher, writer, linguisticProvider, *tzProvider);
                regionIds = regionDumper.Dump(countryIds);
            }

            TSet<ui32> districtIds;
            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TDistrict> writer(MakeFilePath("district"));
                TDistrictDumper districtDumper(fetcher, writer);
                districtIds = districtDumper.Dump(regionIds);
            }

            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TSettlementCode> writer(MakeFilePath("settlement_code"));
                TSettlementCodeDumper settlementCodeDumper(fetcher, writer);
                settlementCodeDumper.Dump();
            }

            TSet<i32> settlementIds;
            THashMap<i32, i32> countryIdToCapitalTimeZoneId;
            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TSettlement> writer(MakeFilePath("settlement"));
                TGeobaseLinguisticProvider linguisticProvider(lookup);
                TSettlementDumper settlementDumper(fetcher, writer, linguisticProvider, *tzProvider);
                auto result = settlementDumper.Dump(districtIds, regionIds, countryIds);
                settlementIds = result.SettlementIds;
                countryIdToCapitalTimeZoneId = result.CountryIdToCapitalTimeZoneId;
            }

            THolder<TRailwayTimezoneProvider> railwayTimezoneProvider;
            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TExpressToCountryLoader expressToCountryLoader(fetcher);
                railwayTimezoneProvider = MakeHolder<TRailwayTimezoneProvider>(
                    expressToCountryLoader.Load(), stationIdToCodes,
                    countryIdToCapitalTimeZoneId, *tzProvider);
            }

            TSet<i32> stationIds;
            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TStation> writer(MakeFilePath("station"));
                TStationDumper stationDumper(fetcher, writer, *tzProvider, *railwayTimezoneProvider);
                stationIds = stationDumper.Dump(settlementIds, districtIds, regionIds, countryIds, supplierIds, stationIdToCodes);
            }

            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TAviaAlliance> writer(MakeFilePath("aviaalliance"));
                TAviaAllianceDumper aviaAllianceDumper(fetcher, writer);
                aviaAllianceDumper.Dump();
            }

            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TCarrier> writer(MakeFilePath("carrier"));
                TCarrierDumper carrierDumper(fetcher, writer);
                carrierDumper.Dump(stationIds, countryIds);
            }

            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TTransportModel> writer(MakeFilePath("transport_model"));
                TVehicleProducerProvider producerProvider(fetcher);
                TTransportModelDumper transportModelDumper(fetcher, writer, producerProvider);
                transportModelDumper.Dump();
            }

            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TStationExpressAlias> writer(MakeFilePath("station_express_alias"));
                TStationExpressAliasDumper stationExpressAliasDumper(fetcher, writer);
                stationExpressAliasDumper.Dump(stationIds);
            }

            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TStation2Settlement> writer(MakeFilePath("settlement2station"));
                TStation2SettlementDumper station2SettlementDumper(fetcher, writer);
                station2SettlementDumper.Dump(stationIds, settlementIds);
            }

            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TTrainTariffInfo> writer(MakeFilePath("train_tariffinfo"));
                TTrainTariffInfoDumper trainTariffInfoDumper(fetcher, writer);
                trainTariffInfoDumper.Dump();
            }

            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TRoute> writer(MakeFilePath("route"));
                TRouteDumper routeDumper(fetcher, writer);
                routeDumper.Dump(supplierIds);
            }

            TThreadLoadingInfo threadLoadingInfo;
            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TThread> writer(MakeFilePath("thread"));
                TThreadDumper threadDumper(fetcher, writer);
                threadLoadingInfo = threadDumper.Dump(stationIds, settlementIds);
            }

            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TThreadStation> writer(MakeFilePath("thread_station"));
                TThreadStationDumper threadStationDumper(fetcher, writer, *tzProvider);
                threadStationDumper.Dump(stationIds, threadLoadingInfo.Ids);
            }

            {
                TMySqlFetcher fetcher(MySqlConfig_);
                TProtobufWriter<TThreadTariff> writer(MakeFilePath("thread_tariff"));
                TThreadTariffDumper threadTariffDumper(fetcher, writer, *tzProvider);
                threadTariffDumper.Dump(threadLoadingInfo.UidToId, stationIds, settlementIds, supplierIds, currencyCodeToId);
            }
        };
    }
}
