#include "region_dumper.h"

using namespace NRasp::NData;

namespace NRasp {
    namespace NDumper {
        TRegionSelectQuery::TRegionSelectQuery()
            : TSelectQuery("www_region")
            , IdField(AddField("id"))

            , TitleDefaultField(AddField("title"))
            , TitleRuField(AddField("title_ru"))
            , TitleUkField(AddField("title_uk"))
            , TitleEnField(AddField("title_en"))
            , TitleTrField(AddField("title_tr"))

            , CountryIdField(AddField("country_id"))
            , TimeZoneField(AddField("time_zone"))
            , GeoIdField(AddField("_geo_id"))

            , IsDisputedTerritoryField(AddField("disputed_territory"))
            , IsHiddenField(AddField("hidden"))

            , AgentGeoIdField(AddField("agent_geo_id"))
            , KladrIdField(AddField("_kladr_id"))
            , KoatuuField(AddField("koatuu"))
        {
        }

        TRegionDumper::TRegionDumper(IFetcher& fetcher, IObjectWriter<TRegion>& writer,
                                     const ILinguisticProvider& linguisticProvider,
                                     const TTimeZoneProvider& timeZoneProvider)
            : Query({})
            , Fetcher_(fetcher)
            , Writer_(writer)
            , LinguisticProvider_(linguisticProvider)
            , TimeZoneProvider_(timeZoneProvider)
        {
        }

        TSet<ui32> TRegionDumper::Dump(const TSet<ui32>& countryIds) {
            INFO_LOG << "Run the region dumper" << Endl;
            Fetcher_.InitQuery(Query.Str());

            TSet<ui32> ids;
            TVector<TString> row(Query.Size());
            while (Fetcher_.GetNext(&row)) {
                TRegion record;

                const auto id = ParseId(row[Query.IdField]);
                const auto geoId = ParseId(row[Query.GeoIdField]);

                record.SetId(id);

                record.SetTitleDefault(row[Query.TitleDefaultField]);
                record.SetTitleRuNominativeCase(GetNominativeCase("ru", row[Query.TitleRuField], geoId));
                record.SetTitleUkNominativeCase(GetNominativeCase("uk", row[Query.TitleUkField], geoId));

                record.MutableTitleNominative()->SetRu(GetNominativeCase("ru", row[Query.TitleRuField], geoId));
                record.MutableTitleNominative()->SetUk(GetNominativeCase("uk", row[Query.TitleUkField], geoId));
                record.MutableTitleNominative()->SetEn(GetNominativeCase("en", row[Query.TitleEnField], geoId));
                record.MutableTitleNominative()->SetTr(GetNominativeCase("tr", row[Query.TitleTrField], geoId));

                {
                    const auto countryId = ParseId(row[Query.CountryIdField]);
                    if (countryId != 0 && !countryIds.contains(countryId)) {
                        WARNING_LOG << "SKIP TRegion: Can not find a country by id: [" << countryId << "] for region with id: [" << id << "]" << Endl;
                        continue;
                    }
                    record.SetCountryId(countryId);
                }
                record.SetTimeZoneId(TimeZoneProvider_.GetTimeZoneId(row[Query.TimeZoneField]));
                record.SetGeoId(geoId);

                record.SetIsDisputedTerritory(ParseBool(row[Query.IsDisputedTerritoryField]));
                record.SetIsHidden(ParseBool(row[Query.IsHiddenField]));

                record.SetAgentGeoId(ParseId(row[Query.AgentGeoIdField]));
                record.SetKladrId(row[Query.KladrIdField]);
                record.SetKoatuu(row[Query.KoatuuField]);

                ids.insert(id);
                Writer_.Write(record);
            };

            if (ids.empty()) {
                ythrow yexception() << "Region dumper has no records.";
            }
            INFO_LOG << "The region dumper has dumped. Total count of regions is " << ids.size() << Endl;

            return ids;
        }

        TString TRegionDumper::GetNominativeCase(const TString& lang, const TString& title, i32 geoId) {
            if (title || !geoId) {
                return title;
            }

            const auto& linguistics = LinguisticProvider_.GetLinguistics(geoId, lang);
            return TString(linguistics.NominativeCase);
        }
    }
}
