#include "region_dumper.h"

#include <travel/rasp/rasp_data/dumper/lib/fetcher/fake_fetcher.h>
#include <travel/rasp/rasp_data/dumper/lib/object_writer/fake_object_writer.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/fabrics.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/fake_linguistic_provider.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/json.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

class TRegionDumperTest: public TTestBase {
private:
    UNIT_TEST_SUITE(TRegionDumperTest);
    UNIT_TEST(TestEmpty);
    UNIT_TEST(TestOneRecord);
    UNIT_TEST(TestUnknownCountry);
    UNIT_TEST(TestEmptyTitleRuNominativeFillFromGeobase);
    UNIT_TEST(TestEmptyTitleUkNominativeFillFromGeobase);
    UNIT_TEST(TestEmptyGeoId);
    UNIT_TEST(TestIsNotDisputedTerritory);
    UNIT_TEST_SUITE_END();

private:
    TFakeFetcher Fetcher_;
    TFakeObjectWriter<TRegion> ObjectWriter_;
    TFakeLinguisticProvider LinguisticProvider_;
    THolder<TTimeZoneProvider> TimeZoneProvider_;
    THolder<TRegionDumper> Dumper_;

    const ui32 DefaultId_ = 999;
    const TString MskTz_ = "Europe/Moscow";
    const ui32 MskTzId_ = 666;
    const ui32 DefaultGeoId_ = 10;
    const ui32 DefaultCountryId_ = 100;

    TSet<ui32> CountryIds_;

public:
    void SetUp() override {
        Fetcher_ = {};
        ObjectWriter_ = TFakeObjectWriter<TRegion>();
        LinguisticProvider_ = {};
        TimeZoneProvider_ = MakeHolder<TTimeZoneProvider>(
            TVector<TTimeZone>{CreateTimeZone(666, MskTz_)},
            MskTz_);
        Dumper_ = MakeHolder<TRegionDumper>(Fetcher_, ObjectWriter_, LinguisticProvider_, *TimeZoneProvider_);

        CountryIds_ = {DefaultCountryId_};
    }

    TVector<TString> CreateRow() {
        return {
            ToString(DefaultId_),        // id
            "title_value",               // title
            "title_ru_value",            // title_ru
            "title_uk_value",            // title_uk
            "title_en_value",            // title_en
            "title_tr_value",            // title_tr
            ToString(DefaultCountryId_), // country_id
            MskTz_,                      // time_zone
            ToString(DefaultGeoId_),     // _geo_id
            "1",                         // disputed_territory
            "0",                         // hidden
            ToString(DefaultGeoId_),     // agent_geo_id
            "kladr_id_value",            // _kladr_id
            "koatuu_value",              // koatuu
        };
    }

    TRegion CreateModel() {
        TRegion record;
        record.SetId(DefaultId_);
        record.SetTitleDefault("title_value");
        record.SetTitleRuNominativeCase("title_ru_value");
        record.SetTitleUkNominativeCase("title_uk_value");
        record.SetCountryId(100);
        record.SetTimeZoneId(MskTzId_);
        record.SetGeoId(DefaultGeoId_);
        record.SetIsDisputedTerritory(true);
        record.SetIsHidden(false);
        record.MutableTitleNominative()->SetRu("title_ru_value");
        record.MutableTitleNominative()->SetUk("title_uk_value");
        record.MutableTitleNominative()->SetEn("title_en_value");
        record.MutableTitleNominative()->SetTr("title_tr_value");
        record.SetAgentGeoId(DefaultGeoId_);
        record.SetKladrId("kladr_id_value");
        record.SetKoatuu("koatuu_value");
        return record;
    }
    void TestEmpty();
    void TestOneRecord();
    void TestUnknownCountry();
    void TestEmptyTitleRuNominativeFillFromGeobase();
    void TestEmptyTitleUkNominativeFillFromGeobase();
    void TestEmptyGeoId();
    void TestIsNotDisputedTerritory();
};

UNIT_TEST_SUITE_REGISTRATION(TRegionDumperTest);

void TRegionDumperTest::TestEmpty() {
    Fetcher_.Add({});

    UNIT_ASSERT_EXCEPTION(Dumper_->Dump(CountryIds_), yexception);
}

void TRegionDumperTest::TestOneRecord() {
    Fetcher_.Add({CreateRow()});

    auto ids = Dumper_->Dump(CountryIds_);
    UNIT_ASSERT_EQUAL(ids, TSet<ui32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TRegion>{CreateModel()}));
}

void TRegionDumperTest::TestUnknownCountry() {
    auto row = CreateRow();
    row[Dumper_->Query.CountryIdField] = "666";
    Fetcher_.Add({row, CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(CountryIds_), TSet<ui32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TRegion>{CreateModel()}));
}

void TRegionDumperTest::TestEmptyTitleRuNominativeFillFromGeobase() {
    auto row = CreateRow();
    row[Dumper_->Query.TitleRuField] = "";

    auto geobaseTitle = "TITLE_NOMINATIVE_FROM_GEOBASE";
    {
        NGeobase::TLinguistics l;
        l.NominativeCase = geobaseTitle;
        LinguisticProvider_.AddLinguistics(l, {DefaultGeoId_, "ru"});
    }
    auto model = CreateModel();
    model.SetTitleRuNominativeCase(geobaseTitle);
    model.MutableTitleNominative()->SetRu(geobaseTitle);

    Fetcher_.Add({row});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(CountryIds_), TSet<ui32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TRegion>{model}));
}

void TRegionDumperTest::TestEmptyTitleUkNominativeFillFromGeobase() {
    auto row = CreateRow();
    row[Dumper_->Query.TitleUkField] = "";

    auto geobaseTitle = "TITLE_NOMINATIVE_FROM_GEOBASE";
    {
        NGeobase::TLinguistics l;
        l.NominativeCase = geobaseTitle;
        LinguisticProvider_.AddLinguistics(l, {DefaultGeoId_, "uk"});
    }
    auto model = CreateModel();
    model.SetTitleUkNominativeCase(geobaseTitle);
    model.MutableTitleNominative()->SetUk(geobaseTitle);

    Fetcher_.Add({row});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(CountryIds_), TSet<ui32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TRegion>{model}));
}

void TRegionDumperTest::TestEmptyGeoId() {
    auto row = CreateRow();
    row[Dumper_->Query.GeoIdField] = "";
    auto model = CreateModel();
    model.SetGeoId(0);

    Fetcher_.Add({row});

    auto ids = Dumper_->Dump(CountryIds_);
    UNIT_ASSERT_EQUAL(ids, TSet<ui32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TRegion>{model}));
}

void TRegionDumperTest::TestIsNotDisputedTerritory() {
    auto row = CreateRow();
    row[Dumper_->Query.IsDisputedTerritoryField] = "0";
    auto model = CreateModel();
    model.SetIsDisputedTerritory(false);

    Fetcher_.Add({row});

    auto ids = Dumper_->Dump(CountryIds_);
    UNIT_ASSERT_EQUAL(ids, TSet<ui32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TRegion>{model}));
}
