#include "settlement_dumper.h"

#include <travel/rasp/rasp_data/dumper/lib/fetcher/mysql_field_parsers.h>

using namespace NRasp::NData;

namespace NRasp {
    namespace NDumper {
        TSettlementSelectQuery::TSettlementSelectQuery()
            : TSelectQuery("www_settlement", false)
            , IdField(AddField("id"))
            , GeoIdField(AddField("_geo_id"))

            , CountryIdField(AddField("country_id"))
            , MajorityIdField(AddField("majority_id"))
            , RegionIdField(AddField("region_id"))
            , DistrictIdField(AddField("district_id"))
            , DisputedTerritoryIdField(AddField("disputed_territory"))
            , TimeZoneField(AddField("time_zone"))

            , IsHiddenField(AddField("hidden"))

            , SlugField(AddField("slug"))

            , LongitudeField(AddField("longitude"))
            , LatitudeField(AddField("latitude"))
            , TitleDefaultField(AddField("title"))

            , TitleRuNominativeField(AddField("title_ru"))
            , TitleRuLocativePrepositionField(AddField("title_ru_preposition_v_vo_na"))
            , TitleRuGenitiveField(AddField("title_ru_genitive"))
            , TitleRuAccusativeField(AddField("title_ru_accusative"))
            , TitleRuPrepositionalField(AddField("title_ru_locative"))

            , TitleUkNominativeField(AddField("title_uk"))
            , TitleEnNominativeField(AddField("title_en"))
            , TitleTrNominativeField(AddField("title_tr"))

            , AbbrTitleDefaultField(AddField("abbr_title"))
            , AbbrTitleRuField(AddField("abbr_title_ru"))
            , AbbrTitleUkField(AddField("abbr_title_uk"))
            , AbbrTitleEnField(AddField("abbr_title_en"))
            , AbbrTitleTrField(AddField("abbr_title_tr"))

            , PhoneInfoField(AddField("phone_info"))
            , PhoneInfoShortField(AddField("phone_info_short"))

            , AgentGeoIdField(AddField("agent_geo_id"))
            , KladrIdField(AddField("_kladr_id"))
            , KoatuuField(AddField("koatuu"))

            , BigCityField(AddField("big_city"))
            , HasTabloField(AddField("has_tablo"))
            , HasManyAirportsField(AddField("has_many_airports"))
            , SuggestOrderField(AddField("suggest_order"))

            , IataField(AddField("iata"))
            , SirenaIdField(AddField("sirena_id"))
        {
        }

        TSettlementDumper::TSettlementDumper(IFetcher& fetcher,
                                             IObjectWriter<TSettlement>& writer,
                                             const ILinguisticProvider& linguisticProvider,
                                             const TTimeZoneProvider& timeZoneProvider)
            : Query({})
            , Fetcher_(fetcher)
            , Writer_(writer)
            , LinguisticProvider_(linguisticProvider)
            , TimeZoneProvider_(timeZoneProvider)
        {
        }

        SettlementDumperResult::SettlementDumperResult(TSet<i32> settlementIds, THashMap<i32, i32> countryIdToCapitalTimeZoneId) {
            SettlementIds = settlementIds;
            CountryIdToCapitalTimeZoneId = countryIdToCapitalTimeZoneId;
        }

        SettlementDumperResult TSettlementDumper::Dump(const TSet<ui32>& districtIds, const TSet<ui32>& regionIds,
                                                       const TSet<ui32>& countryIds) {
            INFO_LOG << "Run the settlement dumper" << Endl;
            Fetcher_.InitQuery(Query.Str());

            TVector<TString> row(Query.Size());
            TSet<i32> settlementIds;
            THashMap<i32, i32> countryIdToCapitalTimeZoneId;
            while (Fetcher_.GetNext(&row)) {
                TSettlement s;
                const auto timeZoneId = TimeZoneProvider_.GetTimeZoneId(row[Query.TimeZoneField]);
                const auto id = ParseId(row[Query.IdField]);

                s.SetId(id);
                s.SetGeoId(ParseId(row[Query.GeoIdField]));
                {
                    auto majorityId = ParseId(row[Query.MajorityIdField]);
                    if (!TSettlement_EMajority_IsValid(majorityId)) {
                        WARNING_LOG << "SKIP TSettlement: Can not find a settlement majority by id: [" << majorityId << "] for settlement with id: [" << id << "]" << Endl;
                        continue;
                    }
                    s.SetMajority(static_cast<TSettlement::EMajority>(majorityId));
                }
                {
                    const auto countryId = ParseId(row[Query.CountryIdField]);
                    if (countryId != 0 && !countryIds.contains(countryId)) {
                        WARNING_LOG << "SKIP TSettlement: Can not find a country by id: [" << countryId << "] for settlement with id: [" << id << "]" << Endl;
                        continue;
                    }
                    s.SetCountryId(countryId);
                }
                {
                    const auto regionId = ParseId(row[Query.RegionIdField]);
                    if (regionId != 0 && regionIds.count(regionId) == 0) {
                        WARNING_LOG << "SKIP TSettlement: Can not find a region by id: [" << regionId << "] for settlement with id: [" << id << "]" << Endl;
                        continue;
                    }
                    s.SetRegionId(regionId);
                }
                {
                    const auto districtId = ParseId(row[Query.DistrictIdField]);
                    if (districtId != 0 && districtIds.count(districtId) == 0) {
                        WARNING_LOG << "SKIP TSettlement: Can not find a district by id: [" << districtId << "] for settlement with id: [" << id << "]" << Endl;
                        continue;
                    }
                    s.SetDistrictId(districtId);
                }

                s.SetIsDisputedTerritory(ParseBool(row[Query.DisputedTerritoryIdField]));
                s.SetTimeZoneId(timeZoneId);

                s.SetIsHidden(ParseBool(row[Query.IsHiddenField]));

                s.SetSlug(row[Query.SlugField]);

                s.SetLongitude(ParseDouble(row[Query.LongitudeField]));
                s.SetLatitude(ParseDouble(row[Query.LatitudeField]));

                s.SetTitleDefault(row[Query.TitleDefaultField]);

                TString titleRuNominative = row[Query.TitleRuNominativeField];
                TString titleRuGenitive = row[Query.TitleRuGenitiveField];
                TString titleRuAccusative = row[Query.TitleRuAccusativeField];
                TString titleRuPrepositional = row[Query.TitleRuPrepositionalField];
                TString titleRuLocativePreposition = row[Query.TitleRuLocativePrepositionField];

                TString titleUkNominative = row[Query.TitleUkNominativeField];
                TString titleEnNominative = row[Query.TitleEnNominativeField];
                TString titleTrNominative = row[Query.TitleTrNominativeField];

                if (s.GetGeoId()) {
                    const auto& ruLings = LinguisticProvider_.GetLinguistics(s.GetGeoId(), "ru");

                    if (titleRuNominative.empty() && !ruLings.NominativeCase.empty()) {
                        titleRuNominative = TString(ruLings.NominativeCase);
                    };
                    if (titleRuGenitive.empty() && !ruLings.GenitiveCase.empty()) {
                        titleRuGenitive = TString(ruLings.GenitiveCase);
                    };
                    if (titleRuAccusative.empty() && !ruLings.AccusativeCase.empty()) {
                        titleRuAccusative = TString(ruLings.AccusativeCase);
                    };
                    if (titleRuLocativePreposition.empty() && !ruLings.Preposition.empty()) {
                        titleRuLocativePreposition = TString(ruLings.Preposition);
                    };
                    if (titleRuPrepositional.empty() && !ruLings.PrepositionalCase.empty()) {
                        titleRuPrepositional = TString(ruLings.PrepositionalCase);
                    };

                    const auto& ukLings = LinguisticProvider_.GetLinguistics(s.GetGeoId(), "uk");
                    if (titleUkNominative.empty() && !ukLings.NominativeCase.empty()) {
                        titleUkNominative = TString(ukLings.NominativeCase);
                    };

                    const auto& trLings = LinguisticProvider_.GetLinguistics(s.GetGeoId(), "tr");
                    if (titleTrNominative.empty() && !trLings.NominativeCase.empty()) {
                        titleTrNominative = TString(trLings.NominativeCase);
                    };

                    const auto& enLings = LinguisticProvider_.GetLinguistics(s.GetGeoId(), "en");
                    if (titleEnNominative.empty() && !enLings.NominativeCase.empty()) {
                        titleEnNominative = TString(enLings.NominativeCase);
                    };
                };

                s.SetTitleRuNominativeCase(titleRuNominative);
                s.SetTitleRuGenitiveCase(titleRuGenitive);
                s.SetTitleRuAccusativeCase(titleRuAccusative);
                s.SetTitleRuPrepositionalCase(titleRuPrepositional);
                s.SetTitleRuPreposition(titleRuLocativePreposition);

                s.SetTitleUkNominativeCase(titleUkNominative);

                s.MutableTitle()->MutableRu()->SetNominative(titleRuNominative);
                s.MutableTitle()->MutableRu()->SetAccusative(titleRuAccusative);
                s.MutableTitle()->MutableRu()->SetPrepositional(titleRuPrepositional);
                s.MutableTitle()->MutableRu()->SetLocativePreposition(titleRuLocativePreposition);
                s.MutableTitle()->MutableRu()->SetGenitive(titleRuGenitive);

                s.MutableTitle()->MutableEn()->SetNominative(titleEnNominative);
                s.MutableTitle()->MutableTr()->SetNominative(titleTrNominative);
                s.MutableTitle()->MutableUk()->SetNominative(titleUkNominative);

                s.SetAbbrTitleDefault(row[Query.AbbrTitleDefaultField]);
                s.MutableAbbrTitle()->SetRu(row[Query.AbbrTitleRuField]);
                s.MutableAbbrTitle()->SetTr(row[Query.AbbrTitleTrField]);
                s.MutableAbbrTitle()->SetUk(row[Query.AbbrTitleUkField]);
                s.MutableAbbrTitle()->SetEn(row[Query.AbbrTitleEnField]);

                s.SetPhoneInfo(row[Query.PhoneInfoField]);
                s.SetPhoneInfoShort(row[Query.PhoneInfoShortField]);

                s.SetAgentGeoId(ParseId(row[Query.AgentGeoIdField]));
                s.SetKladrId(row[Query.KladrIdField]);
                s.SetKoatuu(row[Query.KoatuuField]);

                s.SetBigCity(ParseBool(row[Query.BigCityField]));
                s.SetHasTablo(ParseBool(row[Query.HasTabloField]));
                s.SetHasManyAirports(ParseBool(row[Query.HasManyAirportsField]));
                s.SetSuggestOrder(ParseI32(row[Query.SuggestOrderField]));

                s.SetIata(row[Query.IataField]);
                s.SetSirenaId(row[Query.SirenaIdField]);

                settlementIds.insert(id);

                if (s.GetMajority() == TSettlement::MAJORITY_CAPITAL) {
                    countryIdToCapitalTimeZoneId[s.GetCountryId()] = s.GetTimeZoneId();
                }

                Writer_.Write(s);
            };

            if (settlementIds.empty()) {
                ythrow yexception() << "Settlement dumper has no records.";
            }
            INFO_LOG << "The settlement dumper has dumped. Total count of settlements is " << settlementIds.size() << Endl;

            return SettlementDumperResult(settlementIds, countryIdToCapitalTimeZoneId);
        }
    }
}
