#include "settlement_dumper.h"

#include <travel/rasp/rasp_data/dumper/lib/fetcher/fake_fetcher.h>
#include <travel/rasp/rasp_data/dumper/lib/object_writer/fake_object_writer.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/fabrics.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/fake_linguistic_provider.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/json.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

class TSettlementDumperTest: public TTestBase {
private:
    UNIT_TEST_SUITE(TSettlementDumperTest);
    UNIT_TEST(TestEmpty);
    UNIT_TEST(TestOneRecord);
    UNIT_TEST(TestEmptyGeoId);
    UNIT_TEST(TestUnknownMajority);
    UNIT_TEST(TestUnknownRegion);
    UNIT_TEST(TestUnknownDistrict);
    UNIT_TEST(TestUnknownCountry);
    UNIT_TEST(TestIsNotDisputedTerritory);
    UNIT_TEST(TestEmptyTitleRuNominativeFillFromGeobase);
    UNIT_TEST(TestEmptyTitlePrepositionFillFromGeobase);
    UNIT_TEST(TestEmptyTitleGenitiveFillFromGeobase);
    UNIT_TEST(TestEmptyTitleAccusativeFillFromGeobase);
    UNIT_TEST(TestEmptyTitlePrepositionCaseFillFromGeobase);
    UNIT_TEST(TestEmptyUkNominativeCaseFillFromGeobase);
    UNIT_TEST_SUITE_END();

private:
    TFakeFetcher Fetcher_;
    TFakeObjectWriter<TSettlement> ObjectWriter_;
    TFakeLinguisticProvider LinguisticProvider_;
    THolder<TTimeZoneProvider> TimeZoneProvider_;
    THolder<TSettlementDumper> Dumper_;

    const i32 DefaultId_ = 999;
    const TString MskTz_ = "Europe/Moscow";
    const i32 MskTzId_ = 666;
    const i32 DefaultGeoId_ = 10;
    const ui32 DefaultCountryId_ = 100;
    const ui32 DefaultRegionId_ = 10'000;
    const ui32 DefaultDistrictId_ = 100'000;

    THashMap<i32, i32> CountryIdToCapitalTimeZoneId_;
    TSet<ui32> CountryIds_;
    TSet<ui32> RegionIds_;
    TSet<ui32> DistrictIds_;

public:
    void SetUp() override {
        Fetcher_ = {};
        ObjectWriter_ = TFakeObjectWriter<TSettlement>();
        LinguisticProvider_ = {};
        TimeZoneProvider_ = MakeHolder<TTimeZoneProvider>(
            TVector<TTimeZone>{CreateTimeZone(MskTzId_, MskTz_)},
            MskTz_);
        Dumper_ = MakeHolder<TSettlementDumper>(Fetcher_, ObjectWriter_, LinguisticProvider_, *TimeZoneProvider_);

        CountryIdToCapitalTimeZoneId_ = {{DefaultCountryId_, MskTzId_}};
        CountryIds_ = {DefaultCountryId_};
        RegionIds_ = {DefaultRegionId_};
        DistrictIds_ = {DefaultDistrictId_};
    }

    TVector<TString> CreateRow() {
        return {
            ToString(DefaultId_),                 // id
            ToString(DefaultGeoId_),              // _geo_id
            ToString(DefaultCountryId_),          // country_id
            "1",                                  // majority_id
            ToString(DefaultRegionId_),           // region_id
            ToString(DefaultDistrictId_),         // district_id
            "1",                                  // disputed_territory
            MskTz_,                               // time_zone
            "0",                                  // hidden
            "slug_value",                         // slug
            "1000000.0",                          // longitude
            "10000000.0",                         // latitude
            "title_value",                        // title
            "title_ru_value",                     // title_ru
            "title_ru_preposition_v_vo_na_value", // title_ru_preposition_v_vo_na
            "title_ru_genitive_value",            // title_ru_genitive
            "title_ru_accusative_value",          // title_ru_accusative
            "title_ru_locative_value",            // title_ru_locative
            "title_uk_value",                     // title_uk
            "title_en_value",                     // title_en
            "title_tr_value",                     // title_tr
            "abbr_title_value",                   // abbr_title
            "abbr_title_ru_value",                // abbr_title_ru
            "abbr_title_uk_value",                // abbr_title_uk
            "abbr_title_en_value",                // abbr_title_en
            "abbr_title_tr_value",                // abbr_title_tr
            "phone_info_value",                   // phone_info
            "phone_info_short_value",             // phone_info_short
            ToString(DefaultGeoId_),              // agent_geo_id
            "kladr_id",                           // _kladr_id
            "koatuu",                             // koatuu
            "1",                                  // big_city
            "1",                                  // has_tablo
            "1",                                  // has_many_airports
            "100500",                             // suggest_order
            "QQQ",                                // iata
            "ЯЯЯ",                                // sirena_id
        };
    }

    TSettlement CreateModel() {
        TSettlement s;
        s.SetId(DefaultId_);
        s.SetGeoId(DefaultGeoId_);
        s.SetSlug("slug_value");
        s.SetCountryId(DefaultCountryId_);
        s.SetMajority(TSettlement::MAJORITY_CAPITAL);
        s.SetRegionId(DefaultRegionId_);
        s.SetDistrictId(DefaultDistrictId_);
        s.SetIsDisputedTerritory(true);
        s.SetTimeZoneId(MskTzId_);
        s.SetLongitude(1'000'000.0);
        s.SetLatitude(10'000'000.0);
        s.SetTitleDefault("title_value");
        s.SetTitleRuNominativeCase("title_ru_value");
        s.SetTitleRuGenitiveCase("title_ru_genitive_value");
        s.SetTitleRuAccusativeCase("title_ru_accusative_value");
        s.SetTitleRuPrepositionalCase("title_ru_locative_value");
        s.SetTitleRuPreposition("title_ru_preposition_v_vo_na_value");
        s.SetTitleUkNominativeCase("title_uk_value");

        s.MutableTitle()->MutableRu()->SetNominative("title_ru_value");
        s.MutableTitle()->MutableRu()->SetLocativePreposition("title_ru_preposition_v_vo_na_value");
        s.MutableTitle()->MutableRu()->SetGenitive("title_ru_genitive_value");
        s.MutableTitle()->MutableRu()->SetAccusative("title_ru_accusative_value");
        s.MutableTitle()->MutableRu()->SetPrepositional("title_ru_locative_value");
        s.MutableTitle()->MutableUk()->SetNominative("title_uk_value");
        s.MutableTitle()->MutableEn()->SetNominative("title_en_value");
        s.MutableTitle()->MutableTr()->SetNominative("title_tr_value");

        s.SetAbbrTitleDefault("abbr_title_value");
        s.MutableAbbrTitle()->SetRu("abbr_title_ru_value");
        s.MutableAbbrTitle()->SetUk("abbr_title_uk_value");
        s.MutableAbbrTitle()->SetTr("abbr_title_tr_value");
        s.MutableAbbrTitle()->SetEn("abbr_title_en_value");

        s.SetPhoneInfo("phone_info_value");
        s.SetPhoneInfoShort("phone_info_short_value");

        s.SetKladrId("kladr_id");
        s.SetKoatuu("koatuu");
        s.SetAgentGeoId(DefaultGeoId_);

        s.SetBigCity(true);
        s.SetHasTablo(true);
        s.SetHasManyAirports(true);

        s.SetSuggestOrder(100500);

        s.SetIata("QQQ");
        s.SetSirenaId("ЯЯЯ");

        return s;
    }
    void TestEmpty();
    void TestOneRecord();
    void TestEmptyGeoId();
    void TestUnknownMajority();
    void TestUnknownRegion();
    void TestUnknownDistrict();
    void TestUnknownCountry();
    void TestIsNotDisputedTerritory();
    void TestEmptyTitleRuNominativeFillFromGeobase();
    void TestEmptyTitlePrepositionFillFromGeobase();
    void TestEmptyTitleGenitiveFillFromGeobase();
    void TestEmptyTitleAccusativeFillFromGeobase();
    void TestEmptyTitlePrepositionCaseFillFromGeobase();
    void TestEmptyUkNominativeCaseFillFromGeobase();
};

UNIT_TEST_SUITE_REGISTRATION(TSettlementDumperTest);

void TSettlementDumperTest::TestEmpty() {
    Fetcher_.Add({});

    UNIT_ASSERT_EXCEPTION(Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_), yexception);
}

void TSettlementDumperTest::TestOneRecord() {
    {
        NGeobase::TLinguistics l;
        l.NominativeCase = "SOME_GEOBASE_TITLE";
        l.GenitiveCase = "SOME_GEOBASE_TITLE";
        l.AccusativeCase = "SOME_GEOBASE_TITLE";
        l.Preposition = "SOME_GEOBASE_TITLE";
        l.PrepositionalCase = "SOME_GEOBASE_TITLE";
        LinguisticProvider_.AddLinguistics(l, {DefaultGeoId_, "ru"});
    }
    {
        NGeobase::TLinguistics l;
        l.NominativeCase = "SOME_GEOBASE_TITLE";
        LinguisticProvider_.AddLinguistics(l, {DefaultGeoId_, "uk"});
    }
    Fetcher_.Add({CreateRow()});

    auto result = Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_);
    UNIT_ASSERT_EQUAL(result.SettlementIds, TSet<i32>{DefaultId_});
    UNIT_ASSERT_EQUAL(result.CountryIdToCapitalTimeZoneId, CountryIdToCapitalTimeZoneId_);
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TSettlement>{CreateModel()}));
}

void TSettlementDumperTest::TestEmptyGeoId() {
    auto row = CreateRow();
    row[Dumper_->Query.GeoIdField] = "";
    auto model = CreateModel();
    model.SetGeoId(0);

    Fetcher_.Add({row});

    auto result = Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_);
    UNIT_ASSERT_EQUAL(result.SettlementIds, TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TSettlement>{model}));
}

void TSettlementDumperTest::TestUnknownMajority() {
    auto row = CreateRow();
    row[Dumper_->Query.MajorityIdField] = ToString(TSettlement_EMajority_EMajority_MAX + 1);

    Fetcher_.Add({row, CreateRow()});

    auto result = Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_);
    UNIT_ASSERT_EQUAL(result.SettlementIds, TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TSettlement>{CreateModel()}));
}

void TSettlementDumperTest::TestUnknownRegion() {
    auto row = CreateRow();
    row[Dumper_->Query.RegionIdField] = "666";
    Fetcher_.Add({row, CreateRow()});

    auto result = Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_);
    UNIT_ASSERT_EQUAL(result.SettlementIds, TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TSettlement>{CreateModel()}));
}

void TSettlementDumperTest::TestUnknownDistrict() {
    auto row = CreateRow();
    row[Dumper_->Query.DistrictIdField] = "666";
    Fetcher_.Add({row, CreateRow()});

    auto result = Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_);
    UNIT_ASSERT_EQUAL(result.SettlementIds, TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TSettlement>{CreateModel()}));
}

void TSettlementDumperTest::TestUnknownCountry() {
    auto row = CreateRow();
    row[Dumper_->Query.CountryIdField] = "666";
    Fetcher_.Add({row, CreateRow()});

    auto result = Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_);
    UNIT_ASSERT_EQUAL(result.SettlementIds, TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TSettlement>{CreateModel()}));
}

void TSettlementDumperTest::TestIsNotDisputedTerritory() {
    auto row = CreateRow();
    row[Dumper_->Query.DisputedTerritoryIdField] = "0";
    auto model = CreateModel();
    model.SetIsDisputedTerritory(false);

    Fetcher_.Add({row});

    auto result = Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_);
    UNIT_ASSERT_EQUAL(result.SettlementIds, TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TSettlement>{model}));
}

void TSettlementDumperTest::TestEmptyTitleRuNominativeFillFromGeobase() {
    auto row = CreateRow();
    row[Dumper_->Query.TitleRuNominativeField] = "";

    auto geobaseTitle = "TITLE_NOMINATIVE_FROM_GEOBASE";
    {
        NGeobase::TLinguistics l;
        l.NominativeCase = geobaseTitle;
        LinguisticProvider_.AddLinguistics(l, {DefaultGeoId_, "ru"});
    }
    auto model = CreateModel();
    model.SetTitleRuNominativeCase(geobaseTitle);
    model.MutableTitle()->MutableRu()->SetNominative(geobaseTitle);

    Fetcher_.Add({row});

    auto result = Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_);
    UNIT_ASSERT_EQUAL(result.SettlementIds, TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TSettlement>{model}));
}

void TSettlementDumperTest::TestEmptyTitlePrepositionFillFromGeobase() {
    auto row = CreateRow();
    row[Dumper_->Query.TitleRuLocativePrepositionField] = "";

    auto geobaseTitle = "TITLE_PREPOSITION_FROM_GEOBASE";
    {
        NGeobase::TLinguistics l;
        l.Preposition = geobaseTitle;
        LinguisticProvider_.AddLinguistics(l, {DefaultGeoId_, "ru"});
    }
    auto model = CreateModel();
    model.SetTitleRuPreposition(geobaseTitle);
    model.MutableTitle()->MutableRu()->SetLocativePreposition(geobaseTitle);

    Fetcher_.Add({row});

    auto result = Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_);
    UNIT_ASSERT_EQUAL(result.SettlementIds, TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TSettlement>{model}));
}

void TSettlementDumperTest::TestEmptyTitleGenitiveFillFromGeobase() {
    auto row = CreateRow();
    row[Dumper_->Query.TitleRuGenitiveField] = "";

    auto geobaseTitle = "TITLE_GENITIVE_FROM_GEOBASE";
    {
        NGeobase::TLinguistics l;
        l.GenitiveCase = geobaseTitle;
        LinguisticProvider_.AddLinguistics(l, {DefaultGeoId_, "ru"});
    }
    auto model = CreateModel();
    model.SetTitleRuGenitiveCase(geobaseTitle);
    model.MutableTitle()->MutableRu()->SetGenitive(geobaseTitle);

    Fetcher_.Add({row});

    auto result = Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_);
    UNIT_ASSERT_EQUAL(result.SettlementIds, TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TSettlement>{model}));
}

void TSettlementDumperTest::TestEmptyTitleAccusativeFillFromGeobase() {
    auto row = CreateRow();
    row[Dumper_->Query.TitleRuAccusativeField] = "";

    auto geobaseTitle = "TITLE_ACCUSATIVE_FROM_GEOBASE";
    {
        NGeobase::TLinguistics l;
        l.AccusativeCase = geobaseTitle;
        LinguisticProvider_.AddLinguistics(l, {DefaultGeoId_, "ru"});
    }
    auto model = CreateModel();
    model.SetTitleRuAccusativeCase(geobaseTitle);
    model.MutableTitle()->MutableRu()->SetAccusative(geobaseTitle);

    Fetcher_.Add({row});

    auto result = Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_);
    UNIT_ASSERT_EQUAL(result.SettlementIds, TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TSettlement>{model}));
}

void TSettlementDumperTest::TestEmptyTitlePrepositionCaseFillFromGeobase() {
    auto row = CreateRow();
    row[Dumper_->Query.TitleRuPrepositionalField] = "";

    auto geobaseTitle = "TITLE_PREPOSITION_CASE_FROM_GEOBASE";
    {
        NGeobase::TLinguistics l;
        l.PrepositionalCase = geobaseTitle;
        LinguisticProvider_.AddLinguistics(l, {DefaultGeoId_, "ru"});
    }
    auto model = CreateModel();
    model.SetTitleRuPrepositionalCase(geobaseTitle);
    model.MutableTitle()->MutableRu()->SetPrepositional(geobaseTitle);

    Fetcher_.Add({row});

    auto result = Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_);
    UNIT_ASSERT_EQUAL(result.SettlementIds, TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TSettlement>{model}));
}

void TSettlementDumperTest::TestEmptyUkNominativeCaseFillFromGeobase() {
    auto row = CreateRow();
    row[Dumper_->Query.TitleUkNominativeField] = "";

    auto geobaseTitle = "TITLE_UK_NOMINATIVE_FROM_GEOBASE";
    {
        NGeobase::TLinguistics l;
        l.NominativeCase = geobaseTitle;
        LinguisticProvider_.AddLinguistics(l, {DefaultGeoId_, "uk"});
    }
    auto model = CreateModel();
    model.SetTitleUkNominativeCase(geobaseTitle);
    model.MutableTitle()->MutableUk()->SetNominative(geobaseTitle);

    Fetcher_.Add({row});

    auto result = Dumper_->Dump(DistrictIds_, RegionIds_, CountryIds_);
    UNIT_ASSERT_EQUAL(result.SettlementIds, TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TSettlement>{model}));
}
