#include "station_dumper.h"

using namespace NRasp::NData;

namespace NRasp {
    namespace NDumper {
        TStationSelectQuery::TStationSelectQuery()
            : TSelectQuery("www_station", false)
            , IdField(AddField("id"))
            , SettlementIdField(AddField("settlement_id"))
            , MajorityIdField(AddField("majority_id"))
            , TimeZoneField(AddField("time_zone"))

            , IsHiddenField(AddField("hidden"))

            , LongitudeField(AddField("longitude"))
            , LatitudeField(AddField("latitude"))

            , NotGeneralizeField(AddField("not_generalize"))

            , TitleDefaultField(AddField("title"))

            , RuNominativeTitleField(AddField("title_ru"))
            , RuPrepositionField(AddField("title_ru_preposition_v_vo_na"))
            , RuGenitiveTitleField(AddField("title_ru_genitive"))
            , RuAccusativeTitleField(AddField("title_ru_accusative"))
            , RuPrepositionalTitleField(AddField("title_ru_locative"))
            , UkNominativeTitleField(AddField("title_uk"))

            , PopularTitleDefaultField(AddField("popular_title"))

            , PopularRuNominativeTitleField(AddField("popular_title_ru"))
            , PopularRuGenitiveTitleField(AddField("popular_title_ru_genitive"))

            , PopularUkNominativeTitleField(AddField("popular_title_uk"))

            , TypeIdField(AddField("station_type_id"))
            , TransportTypeIdField(AddField("t_type_id"))

            , RegionIdField(AddField("region_id"))
            , DistrictIdField(AddField("district_id"))
            , CountryIdField(AddField("country_id"))

            , AddressField(AddField("address"))
            , TitleEnField(AddField("title_en"))
            , TitleUkField(AddField("title_uk"))
            , SiteUrlField(AddField("site_url"))
            , HowToGetToCityField(AddField("how_to_get_to_city"))
            , UseDirectionField(AddField("use_direction"))
            , SuburbanZoneIdField(AddField("suburban_zone_id"))
            , HasAeroexpressField(AddField("has_aeroexpress"))
            , NearMetroField(AddField("near_metro"))
            , MapZoomField(AddField("map_zoom"))
            , TypeChoicesField(AddField("type_choices"))
            , LastChangeField(AddField("last_change"))
            , PhotoField(AddField("photo"))
            , PanoramaUrlField(AddField("panorama_url"))
            , ShowSettlementField(AddField("show_settlement"))
            , TabloStateField(AddField("tablo_state"))
            , SupplierIdField(AddField("supplier_id"))
            , PrepositionDepField(AddField("preposition_dep"))
            , FuzzyOnlyField(AddField("fuzzy_only"))
            , VirtualEndField(AddField("virtual_end"))
            , IncompleteBusScheduleField(AddField("incomplete_bus_schedule"))
            , ShowModeField(AddField("show_mode"))
            , IsFuzzyField(AddField("is_fuzzy"))
            , IsSearchableToField(AddField("is_searchable_to"))
            , IsSearchableFromField(AddField("is_searchable_from"))
            , InStationScheduleField(AddField("in_station_schedule"))
            , TitleTrField(AddField("title_tr"))
            , TitleRuOverrideField(AddField("title_ru_override"))
            , TitleEnOverrideField(AddField("title_en_override"))
            , TitleUkOverrideField(AddField("title_uk_override"))
            , TitleTrOverrideField(AddField("title_tr_override"))
            , PopularTitleEnField(AddField("popular_title_en"))
            , PopularTitleTrField(AddField("popular_title_tr"))
            , HowToGetToCityRuField(AddField("how_to_get_to_city_ru"))
            , HowToGetToCityEnField(AddField("how_to_get_to_city_en"))
            , HowToGetToCityUkField(AddField("how_to_get_to_city_uk"))
            , HowToGetToCityTrField(AddField("how_to_get_to_city_tr"))
            , SchemaImageField(AddField("schema_image"))
            , TimeZoneNotCheckField(AddField("time_zone_not_check"))
            , AddressRuField(AddField("address_ru"))
            , AddressEnField(AddField("address_en"))
            , AddressUkField(AddField("address_uk"))
            , AddressTrField(AddField("address_tr"))
            , TabloStatePrevField(AddField("tablo_state_prev"))
            , InThreadField(AddField("in_thread"))
            , ShowTabloStatField(AddField("show_tablo_stat"))
            , IsBaseField(AddField("is_base"))
            , ModifiedAtField(AddField("modified_at"))
            , UseInDepartureForecastField(AddField("use_in_departure_forecast"))
            , UseInForecastField(AddField("use_in_forecast"))
            , IsBaseModifiedAtField(AddField("is_base_modified_at"))
            , SlugField(AddField("slug"))
            , AgreedStopTypeField(AddField("agreed_stop_type"))
        {
        }

        TStationDumper::TStationDumper(
            NRasp::NDumper::IFetcher& fetcher,
            IObjectWriter<NData::TStation>& writer,
            const TTimeZoneProvider& tzProvider,
            const TRailwayTimezoneProvider& railwayTimezoneProvider)
            : Query({})
            , Fetcher_(fetcher)
            , Writer_(writer)
            , TimeZoneProvider_(tzProvider)
            , RailwayTimezoneProvider_(railwayTimezoneProvider)
        {
        }

        TSet<i32>
        TStationDumper::Dump(const TSet<i32>& settlementIds, const TSet<ui32>& districtIds, const TSet<ui32>& regionIds,
                             const TSet<ui32>& countryIds, const TSet<i32>& supplierIds,
                             const THashMap<i32, THashMap<i32, TString>>& stationIdToCodes) {
            INFO_LOG << "Run the station dumper" << Endl;
            Fetcher_.InitQuery(Query.Str());

            TSet<i32> stationIds;
            TVector<TString> row(Query.Size());
            while (Fetcher_.GetNext(&row)) {
                const i32 id = ParseId(row[Query.IdField]);
                const TString timeZoneCode = row[Query.TimeZoneField];
                const auto timeZoneId = TimeZoneProvider_.GetTimeZoneId(timeZoneCode);

                const auto settlementId = ParseId(row[Query.SettlementIdField]);
                if (settlementId != 0 && !settlementIds.contains(settlementId)) {
                    WARNING_LOG << "SKIP TStation: Can not find a settlement by id: [" << settlementId << "] for station with id: [" << id << "]" << Endl;
                    continue;
                }

                const auto majorityId = ParseId(row[Query.MajorityIdField]);
                if (!TStation_EMajority_IsValid(majorityId)) {
                    WARNING_LOG << "SKIP TStation: Can not find a station majority by id: [" << majorityId << "] for station with id: [" << id << "]" << Endl;
                    continue;
                }

                TStation record;
                record.SetId(id);
                record.SetSettlementId(settlementId);
                record.SetMajority(static_cast<TStation::EMajority>(majorityId));
                record.SetTimeZoneId(timeZoneId);
                record.SetTimeZoneCode(timeZoneCode);

                record.SetIsHidden(ParseBool(row[Query.IsHiddenField]));

                record.SetLongitude(ParseDouble(row[Query.LongitudeField]));
                record.SetLatitude(ParseDouble(row[Query.LatitudeField]));

                record.SetNotGeneralize(ParseBool(row[Query.NotGeneralizeField]));

                record.SetTitleDefault((row[Query.TitleDefaultField]));

                record.SetTitleRuNominativeCase(row[Query.RuNominativeTitleField]);
                record.SetTitleRuGenitiveCase(row[Query.RuGenitiveTitleField]);
                record.SetTitleRuAccusativeCase(row[Query.RuAccusativeTitleField]);
                record.SetTitleRuPrepositionalCase(row[Query.RuPrepositionalTitleField]);
                record.SetTitleRuPreposition(row[Query.RuPrepositionField]);

                record.SetTitleUkNominativeCase(row[Query.UkNominativeTitleField]);

                record.SetPopularTitleDefault(row[Query.PopularTitleDefaultField]);

                record.SetPopularTitleRuNominativeCase(row[Query.PopularRuNominativeTitleField]);
                record.SetPopularTitleRuGenitiveCase(row[Query.PopularRuGenitiveTitleField]);
                record.SetPopularTitleUkNominativeCase(row[Query.PopularUkNominativeTitleField]);


                record.MutableAddress()->SetRu(row[Query.AddressRuField]);
                record.MutableAddress()->SetEn(row[Query.AddressEnField]);
                record.MutableAddress()->SetUk(row[Query.AddressUkField]);
                record.MutableAddress()->SetTr(row[Query.AddressTrField]);
                record.SetLocalAddress(row[Query.AddressField]);

                record.MutableTitle()->SetRu(row[Query.RuNominativeTitleField]);
                record.MutableTitle()->SetEn(row[Query.TitleEnField]);
                record.MutableTitle()->SetUk(row[Query.TitleUkField]);
                record.MutableTitle()->SetTr(row[Query.TitleTrField]);
                record.SetLocalTitle(row[Query.TitleDefaultField]);

                record.MutableTitleIn()->SetRu(row[Query.RuPrepositionField] + " " + row[Query.RuPrepositionalTitleField]);
                record.MutableTitleFrom()->SetRu(row[Query.PrepositionDepField] + " " + row[Query.RuGenitiveTitleField]);
                record.MutableTitleTo()->SetRu(row[Query.RuPrepositionField] + " " + row[Query.RuAccusativeTitleField]);

                record.MutablePopularTitle()->SetRu(row[Query.PopularRuNominativeTitleField]);
                record.MutablePopularTitle()->SetEn(row[Query.PopularTitleEnField]);
                record.MutablePopularTitle()->SetUk(row[Query.PopularUkNominativeTitleField]);
                record.MutablePopularTitle()->SetTr(row[Query.PopularTitleTrField]);
                record.SetLocalPopularTitle(row[Query.PopularTitleDefaultField]);

                record.MutableHowToGetToCity()->SetRu(row[Query.HowToGetToCityRuField]);
                record.MutableHowToGetToCity()->SetEn(row[Query.HowToGetToCityEnField]);
                record.MutableHowToGetToCity()->SetUk(row[Query.HowToGetToCityUkField]);
                record.MutableHowToGetToCity()->SetTr(row[Query.HowToGetToCityTrField]);
                record.SetLocalHowToGetToCity(row[Query.HowToGetToCityField]);

                record.MutableShouldOverrideTitle()->SetRu(ParseBool(row[Query.TitleRuOverrideField]));
                record.MutableShouldOverrideTitle()->SetEn(ParseBool(row[Query.TitleEnOverrideField]));
                record.MutableShouldOverrideTitle()->SetUk(ParseBool(row[Query.TitleUkOverrideField]));
                record.MutableShouldOverrideTitle()->SetTr(ParseBool(row[Query.TitleTrOverrideField]));

                record.SetSiteUrl(row[Query.SiteUrlField]);
                record.SetMajorityId(majorityId);
                record.SetHasAeroexpress(ParseBool(row[Query.HasAeroexpressField]));
                record.SetNearMetro(row[Query.NearMetroField]);
                record.SetMapZoom(ParseI32(row[Query.MapZoomField]));
                record.SetTypeChoices(row[Query.TypeChoicesField]);
                record.MutableLastChange()->set_seconds(ParseI32(row[Query.LastChangeField]));
                record.SetPhoto(row[Query.PhotoField]);
                record.SetPanoramaUrl(row[Query.PanoramaUrlField]);
                record.SetShowSettlement(ParseBool(row[Query.ShowSettlementField]));
                record.SetTabloState(row[Query.TabloStateField]);
                record.SetFuzzyOnly(ParseBool(row[Query.FuzzyOnlyField]));
                record.SetVirtualEnd(ParseBool(row[Query.VirtualEndField]));
                record.SetIncompleteBusSchedule(ParseBool(row[Query.IncompleteBusScheduleField]));
                record.SetShowMode(row[Query.ShowModeField]);
                record.SetIsFuzzy(ParseBool(row[Query.IsFuzzyField]));
                record.SetIsSearchableTo(ParseBool(row[Query.IsSearchableToField]));
                record.SetIsSearchableFrom(ParseBool(row[Query.IsSearchableFromField]));
                record.SetInStationSchedule(ParseBool(row[Query.InStationScheduleField]));
                record.SetSchemaImage(row[Query.SchemaImageField]);
                record.SetTimeZoneNotCheck(ParseBool(row[Query.TimeZoneNotCheckField]));
                record.SetTabloStatePrev(row[Query.TabloStatePrevField]);
                record.SetInThread(ParseBool(row[Query.InThreadField]));
                record.SetShowTabloStat(ParseBool(row[Query.ShowTabloStatField]));
                record.SetIsBase(ParseBool(row[Query.IsBaseField]));
                record.MutableModifiedAt()->set_seconds(ParseI32(row[Query.ModifiedAtField]));
                record.SetUseInDepartureForecast(ParseBool(row[Query.UseInDepartureForecastField]));
                record.SetUseInForecast(ParseBool(row[Query.UseInForecastField]));
                record.MutableIsBaseModifiedAt()->set_seconds(ParseI32(row[Query.IsBaseModifiedAtField]));
                record.SetSlug(row[Query.SlugField]);
                record.SetAgreedStopType(row[Query.AgreedStopTypeField]);

                const auto useDirection = row[Query.UseDirectionField];
                if (useDirection.empty()) {
                    record.SetUseDirection(static_cast<TStation::EDirection>(0));
                } else if (useDirection == "dir") {
                    record.SetUseDirection(static_cast<TStation::EDirection>(1));
                } else if (useDirection == "subdir") {
                    record.SetUseDirection(static_cast<TStation::EDirection>(2));
                } else {
                    ythrow yexception() << "unknown useDirection value: " << useDirection;
                }

                auto stationCodes = stationIdToCodes.find(id);
                if (stationCodes != stationIdToCodes.end()) {
                    TStringStream s;
                    for (const auto& kv : stationCodes->second) {
                        s << kv.first << "->" << kv.second << ", ";
                        (*record.MutableStationCodes())[kv.first] = kv.second;
                    }
                    DEBUG_LOG << "Station " << id << " has codes: " << s.Str() << Endl;
                }

                {
                    const auto typeId = ParseId(row[Query.TypeIdField]);
                    if (!TStation_EType_IsValid(typeId)) {
                        WARNING_LOG << "SKIP TStation: Can not find typeId with id [" << typeId << "] for station with id: [" << id << "]" << Endl;
                        continue;
                    }
                    record.SetType(static_cast<TStation::EType>(typeId));
                }

                {
                    const auto transportTypeId = ParseId(row[Query.TransportTypeIdField]);
                    if (!TTransport_EType_IsValid(transportTypeId)) {
                        WARNING_LOG << "SKIP TStation: Can not find transportTypeId with id [" << transportTypeId << "] for station with id: [" << id << "]" << Endl;
                        continue;
                    }
                    record.SetTransportType(
                        static_cast<TTransport::EType>(transportTypeId));
                }

                {
                    const auto capitalTimeZoneId = RailwayTimezoneProvider_.GetTimezoneIdForStation(id);
                    record.SetRailwayTimeZoneId(capitalTimeZoneId);
                }

                {
                    const auto regionId = ParseId(row[Query.RegionIdField]);
                    if (regionId != 0 && !regionIds.contains(regionId)) {
                        WARNING_LOG << "SKIP TStation: Can not find a region by id: [" << regionId << "] for station with id: [" << id << "]" << Endl;
                        continue;
                    }
                    record.SetRegionId(regionId);
                }

                {
                    const auto districtId = ParseId(row[Query.DistrictIdField]);
                    if (districtId != 0 && !districtIds.contains(districtId)) {
                        WARNING_LOG << "SKIP TStation: Can not find a district by id: [" << districtId << "] for station with id: [" << id << "]" << Endl;
                        continue;
                    }
                    record.SetDistrictId(districtId);
                }

                {
                    const auto countryId = ParseId(row[Query.CountryIdField]);
                    if (countryId != 0 && !countryIds.contains(countryId)) {
                        WARNING_LOG << "SKIP TStation: Can not find a country by id: [" << countryId << "] for station with id: [" << id << "]" << Endl;
                        continue;
                    }
                    record.SetCountryId(countryId);
                }

                {
                    const auto supplierId = ParseId(row[Query.SupplierIdField]);
                    if (supplierId != 0 && !supplierIds.contains(supplierId)) {
                        WARNING_LOG << "SKIP TStation: Can not find a supplier by id: [" << supplierId << "] for station with id: [" << id << "]" << Endl;
                        continue;
                    }
                    record.SetSupplierId(supplierId);
                }

                {
                    const auto suburbanZoneId = ParseId(row[Query.SuburbanZoneIdField]);
                    // TODO add suburbanZoneIds to Dump arguments and check that it contains this particular id
                    record.SetSuburbanZoneId(suburbanZoneId);
                }

                stationIds.insert(id);
                Writer_.Write(record);
            };

            if (stationIds.empty()) {
                ythrow yexception() << "Station dumper has no records.";
            }
            INFO_LOG << "The station dumper has dumped. Total count of thread stations is " << stationIds.size() << Endl;

            return stationIds;
        }
    }
}
