#include "station_dumper.h"

#include <travel/rasp/rasp_data/dumper/lib/fetcher/fake_fetcher.h>
#include <travel/rasp/rasp_data/dumper/lib/object_writer/fake_object_writer.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/fabrics.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/json.h>
#include <travel/proto/dicts/rasp/express_to_country.pb.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

class TStationDumperTest: public TTestBase {
private:
    UNIT_TEST_SUITE(TStationDumperTest);
    UNIT_TEST(TestEmpty);
    UNIT_TEST(TestOneRecord);
    UNIT_TEST(TestUnknownSettlement);
    UNIT_TEST(TestEmptySettlement);
    UNIT_TEST(TestUnknownMajority);
    UNIT_TEST(TestUnknownType);
    UNIT_TEST(TestUnknownTransportType);
    UNIT_TEST(TestUnknownRegion);
    UNIT_TEST(TestUnknownDistrict);
    UNIT_TEST(TestUnknownCountry);
    UNIT_TEST(TestRailwayTimeZoneFromExpressToStation);
    UNIT_TEST(TestUnknownRailwayTimeZone);
    UNIT_TEST_SUITE_END();

private:
    TFakeFetcher Fetcher_;
    TFakeObjectWriter<TStation> ObjectWriter_;
    THolder<TTimeZoneProvider> TzProvider_;
    THolder<TRailwayTimezoneProvider> RailwayTimezoneProvider_;
    THolder<TStationDumper> Dumper_;

    const i32 DefaultSettlementId_ = 213;
    const i32 DefaultId_ = 999;
    const ui32 DefaultCountryId_ = 100;
    const ui32 DefaultRegionId_ = 10000;
    const ui32 DefaultDistrictId_ = 100000;
    const i32 DefaultSupplierId_ = 1337;
    const i32 DefaultSuburbanZoneId = 111;

    const TString DefaultTimeZone_ = "Asia/Yekaterinburg";
    const i32 DefaultTimeZoneId_ = 2;

    TSet<i32> SettlementIds_;
    TSet<ui32> DistrictIds_;
    TSet<ui32> RegionIds_;
    TSet<ui32> CountryIds_;
    TSet<i32> SupplierIds_;

    THashMap<i32, THashMap<i32, TString>> StationIdToExpressCode_;
    THashMap<i32, i32> CountryIdToCapitalTimeZoneId_;
    TVector<ExpressToCountryShort> ExpressToCountries_;

public:
    void SetUp() override {
        SettlementIds_ = {DefaultSettlementId_};
        DistrictIds_ = {DefaultDistrictId_};
        RegionIds_ = {DefaultRegionId_};
        CountryIds_ = {DefaultCountryId_};
        SupplierIds_ = {DefaultSupplierId_};
        StationIdToExpressCode_ = {{DefaultId_, {{CODE_SYSTEM_EXPRESS, "1001"}}}, {555, {{CODE_SYSTEM_EXPRESS, "1101"}}}};
        CountryIdToCapitalTimeZoneId_ = {{123, 321}};
        ExpressToCountries_ = {ExpressToCountryShort("10.*", 123, ""), ExpressToCountryShort("11.*", 123, "Asia/Yekaterinburg")};

        Fetcher_ = {};
        ObjectWriter_ = TFakeObjectWriter<TStation>();
        TzProvider_ = MakeHolder<TTimeZoneProvider>(
            TVector<TTimeZone>{
                CreateTimeZone(1, "Europe/Moscow"),
                CreateTimeZone(2, DefaultTimeZone_)},
            "Europe/Moscow");
        RailwayTimezoneProvider_ = MakeHolder<TRailwayTimezoneProvider>(
            ExpressToCountries_,
            StationIdToExpressCode_,
            CountryIdToCapitalTimeZoneId_,
            *TzProvider_);
        Dumper_ = MakeHolder<TStationDumper>(Fetcher_, ObjectWriter_, *TzProvider_, *RailwayTimezoneProvider_);
    }

    TVector<TString> CreateRow() {
        return {
            ToString(DefaultId_),            // id
            ToString(DefaultSettlementId_),  // settlement_id
            "4",                             // majority_id
            DefaultTimeZone_,                // time_zone
            "0",                             // hidden
            "1000000.0",                     // longitude
            "10000000.0",                    // latitude
            "1",                             // not_generalize
            "Москва",                        // title
            "Москва",                        // title_ru
            "В",                             // title_ru_preposition_v_vo_na
            "Москвы",                        // title_ru_genitive
            "Москву",                        // title_ru_accusative
            "Москве",                        // title_ru_locative
            "Москва",                        // title_uk
            "Москва",                        // popular_title
            "Москва",                        // popular_title_ru
            "Москвы",                        // popular_title_ru_genitive
            "Москва",                        // popular_title_uk
            "12",                            // station_type_id
            "3",                             // t_type_id
            ToString(DefaultRegionId_),      // region_id
            ToString(DefaultDistrictId_),    // district_id
            ToString(DefaultCountryId_),     // country_id
            "Москва",                        // address
            "Moscow",                        // title_en
            "Москва",                        // title_uk
            "SiteUrl",                       // site_url
            "Как попасть в Москву",          // how_to_get_to_city
            "dir",                           // use_direction
            ToString(DefaultSuburbanZoneId), // suburban_zone_id
            ToString(true),                  // has_aeroexpress
            "NearMetro",                     // near_metro
            "1",                             // map_zoom
            "TypeChoices",                   // type_choices
            ToString(1591866043),            // last_change
            "Photo",                         // photo
            "PanoramaUrl",                   // panorama_url
            ToString(true),                  // show_settlement
            "TabloState",                    // tablo_state
            ToString(DefaultSupplierId_),    // supplier_id
            "Из",                            // preposition_dep
            ToString(true),                  // fuzzy_only
            ToString(true),                  // virtual_end
            ToString(true),                  // incomplete_bus_schedule
            "ShowMode",                      // show_mode
            ToString(true),                  // is_fuzzy
            ToString(true),                  // is_searchable_to
            ToString(true),                  // is_searchable_from
            ToString(true),                  // in_station_schedule
            "Moskova",                       // title_tr
            ToString(true),                  // title_ru_override
            ToString(true),                  // title_en_override
            ToString(true),                  // title_uk_override
            ToString(true),                  // title_tr_override
            "Moscow",                        // popular_title_en
            "Moskova",                       // popular_title_tr
            "Как попасть в Москву",          // how_to_get_to_city_ru
            "How to get to Moscow",          // how_to_get_to_city_en
            "Як дістатися до Москви",        // how_to_get_to_city_uk
            "Moskova'ya nasıl gidilir",      // how_to_get_to_city_tr
            "SchemaImage",                   // schema_image
            ToString(true),                  // time_zone_not_check
            "Москва",                        // address_ru
            "Moscow",                        // address_en
            "Москва",                        // address_uk
            "Moskova",                       // address_tr
            "TabloStatePrev",                // tablo_state_prev
            ToString(true),                  // in_thread
            ToString(true),                  // show_tablo_stat
            ToString(true),                  // is_base
            ToString(1591866043),            // modified_at
            ToString(true),                  // use_in_departure_forecast
            ToString(true),                  // use_in_forecast
            ToString(1591866043),            // is_base_modified_at
            "Slug",                          // slug
            "AgreedStopType",                // agreed_stop_type
        };
    }
    TStation CreateModel() {
        TStation record;
        record.SetId(DefaultId_);
        record.SetSettlementId(DefaultSettlementId_);
        record.SetMajority(TStation::MAJORITY_STATION);
        record.SetTimeZoneId(DefaultTimeZoneId_);
        record.SetTimeZoneCode(DefaultTimeZone_);
        record.SetLongitude(1'000'000.0);
        record.SetLatitude(10'000'000.0);
        record.SetNotGeneralize(true);
        record.SetType(TStation::TYPE_UNKNOWN_STATION);
        record.SetTransportType(TTransport::TYPE_BUS);
        record.SetRegionId(DefaultRegionId_);
        record.SetDistrictId(DefaultDistrictId_);
        record.SetRailwayTimeZoneId(321);
        record.SetCountryId(DefaultCountryId_);
        record.SetSiteUrl("SiteUrl");
        record.SetMajorityId(4);
        record.SetUseDirection(static_cast<TStation::EDirection>(1));
        record.SetHasAeroexpress(true);
        record.SetNearMetro("NearMetro");
        record.SetMapZoom(1);
        record.SetTypeChoices("TypeChoices");
        record.MutableLastChange()->set_seconds(1591866043);
        record.SetPhoto("Photo");
        record.SetPanoramaUrl("PanoramaUrl");
        record.SetShowSettlement(true);
        record.SetTabloState("TabloState");
        record.SetFuzzyOnly(true);
        record.SetVirtualEnd(true);
        record.SetIncompleteBusSchedule(true);
        record.SetShowMode("ShowMode");
        record.SetIsFuzzy(true);
        record.SetIsSearchableTo(true);
        record.SetIsSearchableFrom(true);
        record.SetInStationSchedule(true);
        record.SetSchemaImage("SchemaImage");
        record.SetTimeZoneNotCheck(true);
        record.SetTabloStatePrev("TabloStatePrev");
        record.SetInThread(true);
        record.SetShowTabloStat(true);
        record.SetIsBase(true);
        record.MutableModifiedAt()->set_seconds(1591866043);
        record.SetUseInDepartureForecast(true);
        record.SetUseInForecast(true);
        record.MutableIsBaseModifiedAt()->set_seconds(1591866043);
        record.SetSlug("Slug");
        record.SetAgreedStopType("AgreedStopType");
        record.SetSupplierId(DefaultSupplierId_);
        record.SetSuburbanZoneId(DefaultSuburbanZoneId);
        record.SetTitleDefault("Москва");
        record.SetTitleRuNominativeCase("Москва");
        record.SetTitleRuGenitiveCase("Москвы");
        record.SetTitleRuAccusativeCase("Москву");
        record.SetTitleRuPrepositionalCase("Москве");
        record.SetTitleRuPreposition("В");
        record.SetTitleUkNominativeCase("Москва");
        record.SetPopularTitleDefault("Москва");
        record.SetPopularTitleRuNominativeCase("Москва");
        record.SetPopularTitleRuGenitiveCase("Москвы");
        record.SetPopularTitleUkNominativeCase("Москва");
        (*record.MutableStationCodes())[CODE_SYSTEM_EXPRESS] = "1001";

        record.MutableAddress()->SetRu("Москва");
        record.MutableAddress()->SetEn("Moscow");
        record.MutableAddress()->SetUk("Москва");
        record.MutableAddress()->SetTr("Moskova");
        record.SetLocalAddress("Москва");

        record.MutableTitle()->SetRu("Москва");
        record.MutableTitle()->SetEn("Moscow");
        record.MutableTitle()->SetUk("Москва");
        record.MutableTitle()->SetTr("Moskova");
        record.SetLocalTitle("Москва");

        record.MutableTitleIn()->SetRu("В Москве");
        record.MutableTitleFrom()->SetRu("Из Москвы");
        record.MutableTitleTo()->SetRu("В Москву");

        record.MutablePopularTitle()->SetRu("Москва");
        record.MutablePopularTitle()->SetEn("Moscow");
        record.MutablePopularTitle()->SetUk("Москва");
        record.MutablePopularTitle()->SetTr("Moskova");
        record.SetLocalPopularTitle("Москва");

        record.MutableHowToGetToCity()->SetRu("Как попасть в Москву");
        record.MutableHowToGetToCity()->SetEn("How to get to Moscow");
        record.MutableHowToGetToCity()->SetUk("Як дістатися до Москви");
        record.MutableHowToGetToCity()->SetTr("Moskova'ya nasıl gidilir");
        record.SetLocalHowToGetToCity("Как попасть в Москву");

        record.MutableShouldOverrideTitle()->SetRu(true);
        record.MutableShouldOverrideTitle()->SetEn(true);
        record.MutableShouldOverrideTitle()->SetUk(true);
        record.MutableShouldOverrideTitle()->SetTr(true);

        return record;
    }
    void TestEmpty();
    void TestOneRecord();
    void TestUnknownSettlement();
    void TestEmptySettlement();
    void TestUnknownMajority();
    void TestUnknownType();
    void TestUnknownTransportType();
    void TestUnknownRegion();
    void TestUnknownDistrict();
    void TestUnknownCountry();
    void TestRailwayTimeZoneFromExpressToStation();
    void TestUnknownRailwayTimeZone();
};

UNIT_TEST_SUITE_REGISTRATION(TStationDumperTest);

void TStationDumperTest::TestEmpty() {
    Fetcher_.Add({});
    UNIT_ASSERT_EXCEPTION(Dumper_->Dump(SettlementIds_, DistrictIds_, RegionIds_, CountryIds_, SupplierIds_, StationIdToExpressCode_).size(), yexception);
}

void TStationDumperTest::TestOneRecord() {
    Fetcher_.Add({CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(SettlementIds_, DistrictIds_, RegionIds_, CountryIds_, SupplierIds_, StationIdToExpressCode_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TStation>{CreateModel()}));
}

void TStationDumperTest::TestUnknownSettlement() {
    auto r = CreateRow();
    r[Dumper_->Query.SettlementIdField] = "666";
    Fetcher_.Add({r, CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(SettlementIds_, DistrictIds_, RegionIds_, CountryIds_, SupplierIds_, StationIdToExpressCode_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TStation>{CreateModel()}));
}

void TStationDumperTest::TestEmptySettlement() {
    auto r = CreateRow();
    r[Dumper_->Query.SettlementIdField] = "";
    Fetcher_.Add({r});

    auto m = CreateModel();
    m.SetSettlementId(0);

    UNIT_ASSERT_EQUAL(Dumper_->Dump(SettlementIds_, DistrictIds_, RegionIds_, CountryIds_, SupplierIds_, StationIdToExpressCode_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TStation>{m}));
}

void TStationDumperTest::TestUnknownMajority() {
    auto r = CreateRow();
    r[Dumper_->Query.MajorityIdField] = ToString(TStation_EMajority_EMajority_MAX + 1);
    Fetcher_.Add({r, CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(SettlementIds_, DistrictIds_, RegionIds_, CountryIds_, SupplierIds_, StationIdToExpressCode_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TStation>{CreateModel()}));
}

void TStationDumperTest::TestUnknownType() {
    auto r = CreateRow();
    r[Dumper_->Query.TypeIdField] = "666";
    Fetcher_.Add({r, CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(SettlementIds_, DistrictIds_, RegionIds_, CountryIds_, SupplierIds_, StationIdToExpressCode_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TStation>{CreateModel()}));
}

void TStationDumperTest::TestUnknownTransportType() {
    auto r = CreateRow();
    r[Dumper_->Query.TransportTypeIdField] = "666";
    Fetcher_.Add({r, CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(SettlementIds_, DistrictIds_, RegionIds_, CountryIds_, SupplierIds_, StationIdToExpressCode_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TStation>{CreateModel()}));
}

void TStationDumperTest::TestUnknownRegion() {
    auto row = CreateRow();
    row[Dumper_->Query.RegionIdField] = "666";
    Fetcher_.Add({row, CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(SettlementIds_, DistrictIds_, RegionIds_, CountryIds_, SupplierIds_, StationIdToExpressCode_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TStation>{CreateModel()}));
}

void TStationDumperTest::TestUnknownDistrict() {
    auto row = CreateRow();
    row[Dumper_->Query.DistrictIdField] = "666";
    Fetcher_.Add({row, CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(SettlementIds_, DistrictIds_, RegionIds_, CountryIds_, SupplierIds_, StationIdToExpressCode_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TStation>{CreateModel()}));
}

void TStationDumperTest::TestUnknownCountry() {
    auto row = CreateRow();
    row[Dumper_->Query.CountryIdField] = "666";
    Fetcher_.Add({row, CreateRow()});

    UNIT_ASSERT_EQUAL(Dumper_->Dump(SettlementIds_, DistrictIds_, RegionIds_, CountryIds_, SupplierIds_, StationIdToExpressCode_), TSet<i32>{DefaultId_});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TStation>{CreateModel()}));
}

void TStationDumperTest::TestRailwayTimeZoneFromExpressToStation() {
    auto r = CreateRow();
    r[Dumper_->Query.IdField] = "555";
    Fetcher_.Add({r});

    auto m = CreateModel();
    m.SetId(555);
    (*m.MutableStationCodes())[CODE_SYSTEM_EXPRESS] = "1101";
    m.SetRailwayTimeZoneId(2);

    UNIT_ASSERT_EQUAL(Dumper_->Dump(SettlementIds_, DistrictIds_, RegionIds_, CountryIds_, SupplierIds_, StationIdToExpressCode_), TSet<i32>{555});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TStation>{m}));
}

void TStationDumperTest::TestUnknownRailwayTimeZone() {
    auto r = CreateRow();
    r[Dumper_->Query.IdField] = "666";
    Fetcher_.Add({r});

    auto m = CreateModel();
    m.SetId(666);
    (*m.MutableStationCodes()).erase(CODE_SYSTEM_EXPRESS);
    m.SetRailwayTimeZoneId(1);

    UNIT_ASSERT_EQUAL(Dumper_->Dump(SettlementIds_, DistrictIds_, RegionIds_, CountryIds_, SupplierIds_, StationIdToExpressCode_), TSet<i32>{666});
    UNIT_ASSERT_STRINGS_EQUAL(ToJson(ObjectWriter_.GetResult()), ToJson(TVector<TStation>{m}));
}
