#include "station_to_settlement_dumper.h"

#include <travel/rasp/rasp_data/dumper/lib/fetcher/fake_fetcher.h>
#include <travel/rasp/rasp_data/dumper/lib/object_writer/fake_object_writer.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/fabrics.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/json.h>
#include <travel/proto/dicts/rasp/timezone.pb.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

class TStation2SettlementDumperTest: public TTestBase {
private:
    UNIT_TEST_SUITE(TStation2SettlementDumperTest);
    UNIT_TEST(TestEmpty);
    UNIT_TEST(TestOneRecord);
    UNIT_TEST(TestUnknownStation);
    UNIT_TEST(TestUnknownSettlement);
    UNIT_TEST(TestDuplicates);
    UNIT_TEST_SUITE_END();

private:
    TSet<i32> SettlementIds_;
    TSet<i32> StationIds_;
    TFakeFetcher Fetcher_;
    TFakeObjectWriter<TStation2Settlement> ObjectWriter_;
    THolder<TStation2SettlementDumper> Dumper_;
    i32 DefaulSettlementId = 213;
    i32 DefaulStationId = 2130;

public:
    void SetUp() override {
        SettlementIds_ = {DefaulSettlementId};
        StationIds_ = {DefaulStationId};
        Fetcher_ = {};
        ObjectWriter_ = TFakeObjectWriter<TStation2Settlement>();

        Dumper_ = MakeHolder<TStation2SettlementDumper>(Fetcher_, ObjectWriter_);
    }

    TVector<TString> CreateRow() {
        return {
            "1",                          // id
            ToString(DefaulStationId),    // station_id
            ToString(DefaulSettlementId), // settlement_id
        };
    }
    TStation2Settlement CreateModel() {
        TStation2Settlement s;
        s.SetStationId(DefaulStationId);
        s.SetSettlementId(DefaulSettlementId);
        return s;
    }
    void TestEmpty();
    void TestOneRecord();
    void TestUnknownStation();
    void TestUnknownSettlement();
    void TestDuplicates();
};

UNIT_TEST_SUITE_REGISTRATION(TStation2SettlementDumperTest);

void TStation2SettlementDumperTest::TestEmpty() {
    Fetcher_.Add({});

    UNIT_ASSERT_EXCEPTION(Dumper_->Dump(StationIds_, SettlementIds_), yexception);
}

void TStation2SettlementDumperTest::TestOneRecord() {
    Fetcher_.Add({CreateRow()});

    Dumper_->Dump(StationIds_, SettlementIds_);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TStation2Settlement>{CreateModel()}));
}

void TStation2SettlementDumperTest::TestUnknownStation() {
    auto r = CreateRow();
    r[Dumper_->Query.StationIdField] = "666";
    Fetcher_.Add({r, CreateRow()});

    Dumper_->Dump(StationIds_, SettlementIds_);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TStation2Settlement>{CreateModel()}));
}

void TStation2SettlementDumperTest::TestUnknownSettlement() {
    auto r = CreateRow();
    r[Dumper_->Query.SettlementIdField] = "666";
    Fetcher_.Add({r, CreateRow()});

    Dumper_->Dump(StationIds_, SettlementIds_);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TStation2Settlement>{CreateModel()}));
}

void TStation2SettlementDumperTest::TestDuplicates() {
    Fetcher_.Add({CreateRow()});
    Fetcher_.Add({CreateRow()});

    Dumper_->Dump(StationIds_, SettlementIds_);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TStation2Settlement>{CreateModel()}));
}
