#include "thread_dumper.h"

#include <travel/rasp/rasp_data/dumper/lib/fetcher/fake_fetcher.h>
#include <travel/rasp/rasp_data/dumper/lib/object_writer/fake_object_writer.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/json.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

Y_UNIT_TEST_SUITE(ThreadSelectQuery) {
    Y_UNIT_TEST(Test) {
        TString someId = "123";
        auto res = TThreadSelectQuery();
        UNIT_ASSERT_STRINGS_EQUAL(
            res.Str(),
            "SELECT id, hidden, tz_year_days, tz_start_time, type_id, title_common, route_number, t_type_id, uid, route_id, company_id FROM www_rthread;");
    }
}

class TThreadDumperTest: public TTestBase {
private:
    UNIT_TEST_SUITE(TThreadDumperTest);
    UNIT_TEST(TestEmpty);
    UNIT_TEST(TestOneRecord);
    UNIT_TEST(TestUnknownType);
    UNIT_TEST(TestUnknownTransportType);
    UNIT_TEST_SUITE_END();

private:
    TFakeFetcher Fetcher_;
    TFakeObjectWriter<TThread> ObjectWriter_;
    THolder<TThreadDumper> Dumper_;
    i32 mskId = 213;
    i32 mskStationId = 2130;
    TSet<i32> StationIds_;
    TSet<i32> SettlementIds_;
    TVector<TString> DefaultRow_;
    TThread DefaultModel_;

public:
    void SetUp() override {
        Fetcher_ = {};
        ObjectWriter_ = TFakeObjectWriter<TThread>();
        Dumper_ = MakeHolder<TThreadDumper>(Fetcher_, ObjectWriter_);
        StationIds_ = {100};
        SettlementIds_ = {10};
        DefaultRow_ = CreateRow();
        DefaultModel_ = CreateModel();
    }

    TVector<TString> CreateRow() {
        return {
            "100",                                                        // id
            "0",                                                          // hidden
            TString(12 * 31, '1'),                                   // tz_year_days
            "12:30:10",                                                   // tz_start_time
            "10",                                                         // type_id
            "{\"add_ring\": true, \"title_parts\": [\"s100\", \"c10\"]}", // title_common
            "su 123",                                                     // route_number
            "7",                                                          // t_type_id
            "uid_value",                                                  // uid
            "1000",                                                       // route_id
            "112",                                                        // company_id
        };
    }
    TThread CreateModel() {
        TThread s;
        s.SetId(100);
        s.SetTzStartTime((12 * 60 + 30) * 60 + 10);
        s.SetType(TThread::TYPE_CANCEL);
        s.SetNumber("su 123");
        auto& yearsDays = *s.MutableYearDays();
        for (int i = 0; i < 12; ++i) {
            yearsDays.Add(2147483647);
        }
        s.SetTransportType(TTransport::TYPE_HELICOPTER);
        auto& titleCommon = *s.MutableCommonTitle();
        titleCommon.SetIsRing(true);
        titleCommon.SetIsCombined(false);
        auto& parts = *titleCommon.MutableTitleParts();
        auto& stationPart = *parts.Add();
        stationPart.SetStationId(100);
        auto& setttlementPart = *parts.Add();
        setttlementPart.SetSettlementId(10);
        s.SetUid("uid_value");
        s.SetRouteId(1000);
        s.SetCompanyId(112);

        return s;
    }
    void TestEmpty();
    void TestOneRecord();
    void TestUnknownType();
    void TestUnknownTransportType();
};

UNIT_TEST_SUITE_REGISTRATION(TThreadDumperTest);

void TThreadDumperTest::TestEmpty() {
    Fetcher_.Add({});

    UNIT_ASSERT_EXCEPTION(Dumper_->Dump(StationIds_, SettlementIds_), yexception);
}

void TThreadDumperTest::TestOneRecord() {
    Fetcher_.Add({DefaultRow_});

    auto info = Dumper_->Dump(StationIds_, SettlementIds_);
    UNIT_ASSERT_EQUAL(info.Ids, TSet<i32>{100});
    {
        TMap<TString, i32> expected;
        expected["uid_value"] = 100;
        UNIT_ASSERT_EQUAL(info.UidToId, expected);
    }
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThread>{DefaultModel_}));
}

void TThreadDumperTest::TestUnknownType() {
    auto row = CreateRow();
    row[Dumper_->Query.TypeIdField] = "666";
    Fetcher_.Add({row, DefaultRow_});

    auto info = Dumper_->Dump(StationIds_, SettlementIds_);
    UNIT_ASSERT_EQUAL(info.Ids, TSet<i32>{100});
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThread>{DefaultModel_}));
}

void TThreadDumperTest::TestUnknownTransportType() {
    auto row = CreateRow();
    row[Dumper_->Query.TransportTypeIdField] = "666";
    Fetcher_.Add({row, DefaultRow_});

    auto info = Dumper_->Dump(StationIds_, SettlementIds_);
    UNIT_ASSERT_EQUAL(info.Ids, TSet<i32>{100});
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThread>{DefaultModel_}));
}
