#include "thread_station_dumper.h"

#include <travel/rasp/rasp_data/dumper/lib/fetcher/mysql_field_parsers.h>

using namespace NRasp::NData;

namespace NRasp {
    namespace NDumper {
        TThreadStationSelectQuery::TThreadStationSelectQuery()
            : TSelectQuery("www_rtstation", false)
            , IdField(AddField("id"))
            , StationIdField(AddField("station_id"))
            , TzDepartureField(AddField("tz_departure"))
            , TzArrivalField(AddField("tz_arrival"))
            , TimeZoneField(AddField("time_zone"))
            , IsSearchableFromField(AddField("is_searchable_from"))
            , IsSearchableToField(AddField("is_searchable_to"))
            , DepartureCodeSharingField(AddField("departure_code_sharing"))
            , ArrivalCodeSharingField(AddField("arrival_code_sharing"))
            , ThreadIdField(AddField("thread_id"))
            , IsTechnicalStopField(AddField("is_technical_stop"))
        {
        }

        TThreadStationDumper::TThreadStationDumper(IFetcher& fetcher,
                                                   IObjectWriter<TThreadStation>& writer,
                                                   const TTimeZoneProvider& timeZoneProvider)
            : Query({})
            , Fetcher_(fetcher)
            , Writer_(writer)
            , TimeZoneProvider_(timeZoneProvider)
        {
        }

        void TThreadStationDumper::Dump(const TSet<i32>& stationIds, const TSet<i32>& threadIds) {
            INFO_LOG << "Run the thread station dumper" << Endl;
            Fetcher_.InitQuery(Query.Str());

            size_t threadStationCount = 0;
            TVector<TString> row(Query.Size());
            while (Fetcher_.GetNext(&row)) {
                const auto id = ParseId(row[Query.IdField]);
                const auto tzDeparture = ParseOptionalI32(row[Query.TzDepartureField]);
                const auto tzArrival = ParseOptionalI32(row[Query.TzArrivalField]);

                auto stationId = ParseId(row[Query.StationIdField]);
                if (stationIds.count(stationId) == 0) {
                    WARNING_LOG << "SKIP TThreadStation: Can not find station with id [" << stationId << "] for thread station with id: [" << id << "]" << Endl;
                    continue;
                }
                auto threadId = ParseId(row[Query.ThreadIdField]);
                if (threadIds.count(threadId) == 0) {
                    WARNING_LOG << "SKIP TThreadStation: Can not find thread with id [" << threadId << "] for thread station with id: [" << id << "]" << Endl;
                    continue;
                }
                ++threadStationCount;

                TThreadStation record;
                record.SetId(id);
                record.SetStationId(stationId);

                if (tzDeparture.Defined()) {
                    record.SetHasDeparture(true);
                    record.SetDepartureTz(tzDeparture.GetRef());
                }

                if (tzArrival.Defined()) {
                    record.SetHasArrival(true);
                    record.SetArrivalTz(tzArrival.GetRef());
                }

                record.SetTimeZoneId(TimeZoneProvider_.GetTimeZoneId(row[Query.TimeZoneField]));
                record.SetIsSearchableFrom(ParseBool(row[Query.IsSearchableFromField]));
                record.SetIsSearchableTo(ParseBool(row[Query.IsSearchableToField]));
                record.SetIsDepartureCodeSharing(ParseBool(row[Query.DepartureCodeSharingField]));
                record.SetIsArrivalCodeSharing(ParseBool(row[Query.ArrivalCodeSharingField]));
                record.SetThreadId(threadId);
                record.SetIsTechnicalStop(ParseBool(row[Query.IsTechnicalStopField]));

                Writer_.Write(record);
            }

            if (!threadStationCount) {
                ythrow yexception() << "Thread stations dumper has no records.";
            }
            INFO_LOG << "Total count of thread stations is " << threadStationCount << Endl;
        }
    }
}
