#include "thread_station_dumper.h"

#include <travel/rasp/rasp_data/dumper/lib/tester/json.h>
#include <travel/rasp/rasp_data/dumper/lib/object_writer/fake_object_writer.h>
#include <travel/rasp/rasp_data/dumper/lib/fetcher/fake_fetcher.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/fabrics.h>

#include <library/cpp/testing/unittest/registar.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

class TThreadStationDumperTest: public TTestBase {
private:
    UNIT_TEST_SUITE(TThreadStationDumperTest);
    UNIT_TEST(TestEmpty);
    UNIT_TEST(TestOneRecord);
    UNIT_TEST(TestUnknownStation);
    UNIT_TEST(TestUnknownThread);
    UNIT_TEST(TestIsSearchableFrom);
    UNIT_TEST(TestIsSearchableTo);
    UNIT_TEST(TestDepartureCodeSharing);
    UNIT_TEST(TestArrivalCodeSharing);
    UNIT_TEST(TestIsTechnicalStop);
    UNIT_TEST(TestWithoutDeparture);
    UNIT_TEST(TestWithoutArrival);
    UNIT_TEST_SUITE_END();

private:
    TFakeFetcher Fetcher_;
    TFakeObjectWriter<TThreadStation> ObjectWriter_;
    THolder<TTimeZoneProvider> TimeZoneProvider_;
    THolder<TThreadStationDumper> Dumper_;
    i32 DefaultStationId_;
    i32 DefaultThreadId_;
    TString DefaultTimeZoneCode_;
    i32 DefaultTimeZoneId_;
    TSet<i32> StationIds_;
    TSet<i32> ThreadIds_;

public:
    void SetUp() override {
        DefaultStationId_ = 1000;
        DefaultThreadId_ = 200;
        DefaultTimeZoneCode_ = "Asia/Vladivostok";
        DefaultTimeZoneId_ = 40;
        StationIds_ = {DefaultStationId_};
        ThreadIds_ = {DefaultThreadId_};

        Fetcher_ = {};
        ObjectWriter_ = TFakeObjectWriter<TThreadStation>();
        TimeZoneProvider_ = MakeHolder<TTimeZoneProvider>(
            TVector<TTimeZone>{
                CreateTimeZone(666, "Europe/Moscow"),
                CreateTimeZone(DefaultTimeZoneId_, DefaultTimeZoneCode_)},
            "Europe/Moscow");

        Dumper_ = MakeHolder<TThreadStationDumper>(Fetcher_, ObjectWriter_, *TimeZoneProvider_);
    }

    TVector<TString> CreateRow() {
        return {
            "100",                       //  id
            ToString(DefaultStationId_), //  station_id
            "3000",                      //  tz_departure
            "-3000",                     //  tz_arrival
            DefaultTimeZoneCode_,        //  time_zone
            "1",                         //  is_searchable_from
            "1",                         //  is_searchable_to
            "1",                         //  departure_code_sharing
            "1",                         //  arrival_code_sharing
            ToString(DefaultThreadId_),  //  thread_id
            "1",                         //  is_technical_stop
        };
    }
    TThreadStation CreateModel() {
        TThreadStation s;
        s.SetId(100);
        s.SetStationId(DefaultStationId_);
        s.SetDepartureTz(3000);
        s.SetArrivalTz(-3000);
        s.SetHasDeparture(true);
        s.SetHasArrival(true);
        s.SetTimeZoneId(DefaultTimeZoneId_);
        s.SetIsSearchableFrom(true);
        s.SetIsSearchableTo(true);
        s.SetIsDepartureCodeSharing(true);
        s.SetIsArrivalCodeSharing(true);
        s.SetThreadId(DefaultThreadId_);
        s.SetIsTechnicalStop(200);

        return s;
    }
    void TestEmpty();
    void TestOneRecord();
    void TestUnknownStation();
    void TestUnknownThread();
    void TestIsSearchableFrom();
    void TestIsSearchableTo();
    void TestDepartureCodeSharing();
    void TestArrivalCodeSharing();
    void TestIsTechnicalStop();
    void TestWithoutDeparture();
    void TestWithoutArrival();
};

UNIT_TEST_SUITE_REGISTRATION(TThreadStationDumperTest);

void TThreadStationDumperTest::TestEmpty() {
    Fetcher_.Add({});

    UNIT_ASSERT_EXCEPTION(Dumper_->Dump(StationIds_, ThreadIds_), yexception);
}

void TThreadStationDumperTest::TestOneRecord() {
    Fetcher_.Add({CreateRow()});

    Dumper_->Dump(StationIds_, ThreadIds_);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThreadStation>{CreateModel()}));
}

void TThreadStationDumperTest::TestUnknownStation() {
    auto row = CreateRow();
    row[Dumper_->Query.StationIdField] = "";
    Fetcher_.Add({row, CreateRow()});

    Dumper_->Dump(StationIds_, ThreadIds_);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThreadStation>{CreateModel()}));
}

void TThreadStationDumperTest::TestUnknownThread() {
    auto row = CreateRow();
    row[Dumper_->Query.ThreadIdField] = "";
    Fetcher_.Add({row, CreateRow()});

    Dumper_->Dump(StationIds_, ThreadIds_);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThreadStation>{CreateModel()}));
}

void TThreadStationDumperTest::TestIsSearchableFrom() {
    auto row = CreateRow();
    row[Dumper_->Query.IsSearchableFromField] = "0";
    Fetcher_.Add({row});

    Dumper_->Dump(StationIds_, ThreadIds_);

    auto model = CreateModel();
    model.SetIsSearchableFrom(false);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThreadStation>{model}));
}

void TThreadStationDumperTest::TestIsSearchableTo() {
    auto row = CreateRow();
    row[Dumper_->Query.IsSearchableToField] = "0";
    Fetcher_.Add({row});

    Dumper_->Dump(StationIds_, ThreadIds_);

    auto model = CreateModel();
    model.SetIsSearchableTo(false);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThreadStation>{model}));
}

void TThreadStationDumperTest::TestDepartureCodeSharing() {
    auto row = CreateRow();
    row[Dumper_->Query.DepartureCodeSharingField] = "0";
    Fetcher_.Add({row});

    Dumper_->Dump(StationIds_, ThreadIds_);

    auto model = CreateModel();
    model.SetIsDepartureCodeSharing(false);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThreadStation>{model}));
}

void TThreadStationDumperTest::TestArrivalCodeSharing() {
    auto row = CreateRow();
    row[Dumper_->Query.ArrivalCodeSharingField] = "0";
    Fetcher_.Add({row});

    Dumper_->Dump(StationIds_, ThreadIds_);

    auto model = CreateModel();
    model.SetIsArrivalCodeSharing(false);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThreadStation>{model}));
}

void TThreadStationDumperTest::TestIsTechnicalStop() {
    auto row = CreateRow();
    row[Dumper_->Query.IsTechnicalStopField] = "0";
    Fetcher_.Add({row});

    Dumper_->Dump(StationIds_, ThreadIds_);

    auto model = CreateModel();
    model.SetIsTechnicalStop(false);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThreadStation>{model}));
}

void TThreadStationDumperTest::TestWithoutDeparture() {
    auto row = CreateRow();
    row[Dumper_->Query.TzDepartureField] = "NULL";
    Fetcher_.Add({row});

    Dumper_->Dump(StationIds_, ThreadIds_);

    auto model = CreateModel();
    model.SetHasDeparture(false);
    model.SetDepartureTz(0);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThreadStation>{model}));
}

void TThreadStationDumperTest::TestWithoutArrival() {
    auto row = CreateRow();
    row[Dumper_->Query.TzArrivalField] = "NULL";
    Fetcher_.Add({row});

    Dumper_->Dump(StationIds_, ThreadIds_);

    auto model = CreateModel();
    model.SetHasArrival(false);
    model.SetArrivalTz(0);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThreadStation>{model}));
}
