#include "thread_tariff_dumper.h"

#include <travel/rasp/rasp_data/dumper/lib/year_mask/year_mask.h>
#include <travel/rasp/rasp_data/dumper/lib/fetcher/mysql_field_parsers.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

namespace NRasp {
    namespace NDumper {
        TThreadTariffSelectQuery::TThreadTariffSelectQuery()
            : TSelectQuery("www_threadtariff", false)
            , IdField(AddField("id"))
            , ThreadlUidField(AddField("thread_uid"))
            , StationFromIdField(AddField("station_from_id"))
            , StationToIdField(AddField("station_to_id"))
            , TariffField(AddField("tariff"))
            , YearsDaysFromField(AddField("year_days_from"))
            , SettlmenentFromIdField(AddField("settlement_from_id"))
            , SettlmenentToIdField(AddField("settlement_to_id"))
            , CurrencyField(AddField("currency"))
            , TimeZoneFromField(AddField("time_zone_from"))
            , TTypeIdField(AddField("t_type_id"))
            , SupplierIdField(AddField("supplier_id"))
            , TimeFromField(AddField("time_from"))
        {
        }
        TThreadTariffDumper::TThreadTariffDumper(IFetcher& fetcher,
                                                 IObjectWriter<TThreadTariff>& writer,
                                                 TTimeZoneProvider& timeZoneProvider)
            : Query({})
            , Fetcher_(fetcher)
            , Writer_(writer)
            , TimeZoneProvider_(timeZoneProvider)
        {
        }

        void TThreadTariffDumper::Dump(const TMap<TString, i32>& threadUidToId, const TSet<i32>& stationIds, const TSet<i32>& settlementIds, const TSet<i32>& supplierIds, const TMap<TString, i32>& currencyCodeToId) {
            INFO_LOG << "Run the thread tariff dumper" << Endl;
            Fetcher_.InitQuery(Query.Str());

            size_t threadTariffsCount = 0;
            TVector<TString> row(Query.Size());
            while (Fetcher_.GetNext(&row)) {
                TThreadTariff record;
                i32 id = ParseId(row[Query.IdField]);

                TString uid(row[Query.ThreadlUidField]);
                auto threadIt = threadUidToId.find(uid);
                if (threadIt == threadUidToId.end()) {
                    WARNING_LOG << "SKIP TThreadTariff: Can not find thread with uid [" << uid << "] for thread tariff with id: [" << id << "]" << Endl;
                    continue;
                }

                i32 stationFromId(ParseId(row[Query.StationFromIdField]));
                if (stationFromId != 0 && stationIds.count(stationFromId) == 0) {
                    WARNING_LOG << "SKIP TThreadTariff: Can not find statoinFromId with id [" << stationFromId << "] for thread tariff with id: [" << id << "]" << Endl;
                    continue;
                }
                i32 stationToId(ParseId(row[Query.StationToIdField]));
                if (stationToId != 0 && stationIds.count(stationToId) == 0) {
                    WARNING_LOG << "SKIP TThreadTariff: Can not find statoinToId with id [" << stationToId << "] for thread tariff with id: [" << id << "]" << Endl;
                    continue;
                }

                i32 settlementFromId(ParseId(row[Query.SettlmenentFromIdField]));
                if (settlementFromId != 0 && settlementIds.count(settlementFromId) == 0) {
                    WARNING_LOG << "SKIP TThreadTariff: Can not find settlementFromId with id [" << settlementFromId << "] for thread tariff with id: [" << id << "]" << Endl;
                    continue;
                }
                i32 settlementToId(ParseId(row[Query.SettlmenentToIdField]));
                if (settlementToId != 0 && settlementIds.count(settlementToId) == 0) {
                    WARNING_LOG << "SKIP TThreadTariff: Can not find settlementToId with id [" << settlementToId << "] for thread tariff with id: [" << id << "]" << Endl;
                    continue;
                }

                record.SetId(id);
                record.SetThreadId(threadIt->second);
                record.SetFromStationId(stationFromId);
                record.SetToStationId(stationToId);
                record.SetTariff(ParseDouble(row[Query.TariffField]));
                auto mask = ParseYearMask(row[Query.YearsDaysFromField]);
                (*record.MutableYearDaysFrom()) = ::google::protobuf::RepeatedField<i32>(mask.begin(), mask.end());
                record.SetFromSettlementId(settlementFromId);
                record.SetToSettlementId(settlementToId);
                {
                    const auto& currencyCode = row[Query.CurrencyField];
                    auto currencyIt = currencyCodeToId.find(currencyCode);
                    if (currencyIt == currencyCodeToId.end()) {
                        WARNING_LOG << "SKIP TThreadTariff: Can not find currencyCode with code [" << currencyCode << "] for thread tariff with id: [" << id << "]" << Endl;
                        continue;
                    }
                    record.SetCurrencyId(currencyIt->second);
                }
                record.SetFromTimeZoneId(TimeZoneProvider_.GetTimeZoneId(row[Query.TimeZoneFromField]));
                record.SetTransportType(static_cast<TTransport::EType>(ParseId(row[Query.TTypeIdField])));
                {
                    auto supplierId = ParseId(row[Query.SupplierIdField]);
                    if (supplierIds.count(supplierId) == 0) {
                        WARNING_LOG << "SKIP TThreadTariff: Can not find supplier with id [" << supplierId << "] for thread tariff with id: [" << id << "]" << Endl;
                        continue;
                    }
                    record.SetSupplierId(supplierId);
                }
                record.SetFromTime(ParseTime(row[Query.TimeFromField]));

                ++threadTariffsCount;
                Writer_.Write(record);
            };

            if (!threadTariffsCount) {
                ythrow yexception() << "Thread tariff dumper has no records.";
            }
            INFO_LOG << "Total count of thread tariffs is " << threadTariffsCount << Endl;
        }
    }
}
