#include "thread_tariff_dumper.h"

#include <travel/rasp/rasp_data/dumper/lib/tester/json.h>
#include <travel/rasp/rasp_data/dumper/lib/object_writer/fake_object_writer.h>
#include <travel/rasp/rasp_data/dumper/lib/fetcher/fake_fetcher.h>
#include <travel/rasp/rasp_data/dumper/lib/timezone_provider/timezone_provider.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/fabrics.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/json/json_reader.h>

#include <util/generic/list.h>
#include <util/generic/vector.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

class TThreadTariffDumperTest: public TTestBase {
private:
    UNIT_TEST_SUITE(TThreadTariffDumperTest);
    UNIT_TEST(TestEmpty);
    UNIT_TEST(TestOneRecord);
    UNIT_TEST_SUITE_END();

private:
    i32 DefaultFromStationId_;
    i32 DefaultToStationId_;
    i32 DefaultFromSettlementId_;
    i32 DefaultToSettlementId_;
    TString DefaultThreadUid_;
    i32 DefaultThreadId_;
    TString DefaultTimeZoneCode_;
    i32 DefaultTimeZoneId_;
    TString DefaultCurrencyCode_;
    i32 DefaultCurrencyId_;
    i32 DefaultSupplierId_;

    TMap<TString, i32> ThreadUidToId_;
    TSet<i32> StationIds_;
    TSet<i32> SettlementIds_;
    TSet<i32> SupplierIds_;
    TMap<TString, i32> CurrencyCodeToId_;

    TMap<TString, i32> currencyCodeToId;
    TFakeFetcher Fetcher_;
    TFakeObjectWriter<TThreadTariff> ObjectWriter_;
    THolder<TTimeZoneProvider> TimeZoneProvider_;
    THolder<TThreadTariffDumper> Dumper_;

public:
    void SetUp() override {
        DefaultFromStationId_ = 10;
        DefaultToStationId_ = 100;
        DefaultFromSettlementId_ = 1'000;
        DefaultToSettlementId_ = 10'000;
        DefaultThreadUid_ = "thread_uid_value";
        DefaultThreadId_ = 999;
        DefaultTimeZoneCode_ = "Asia/Vladivostok";
        DefaultTimeZoneId_ = 555;
        DefaultCurrencyCode_ = "RUB";
        DefaultCurrencyId_ = 777;
        DefaultSupplierId_ = 100'000;

        ThreadUidToId_ = {{DefaultThreadUid_, DefaultThreadId_}};
        StationIds_ = {DefaultFromStationId_, DefaultToStationId_};
        SettlementIds_ = {DefaultFromSettlementId_, DefaultToSettlementId_};
        SupplierIds_ = {DefaultSupplierId_};
        CurrencyCodeToId_ = {{DefaultCurrencyCode_, DefaultCurrencyId_}};

        Fetcher_ = {};
        ObjectWriter_ = TFakeObjectWriter<TThreadTariff>();
        TimeZoneProvider_ = MakeHolder<TTimeZoneProvider>(
            TVector<TTimeZone>{
                CreateTimeZone(666, "Europe/Moscow"),
                CreateTimeZone(DefaultTimeZoneId_, DefaultTimeZoneCode_)},
            "Europe/Moscow");

        Dumper_ = MakeHolder<TThreadTariffDumper>(Fetcher_, ObjectWriter_, *TimeZoneProvider_);
    }

    TVector<TString> CreateRow() {
        return {
            "1",                                //  id
            DefaultThreadUid_,                  //  thread_uid
            ToString(DefaultFromStationId_),    //  station_from_id
            ToString(DefaultToStationId_),      //  station_to_id
            "222.0",                            //  tariff
            TString(12 * 31, '1'),              //  year_days_from
            ToString(DefaultFromSettlementId_), //  settlement_from_id
            ToString(DefaultToSettlementId_),   //  settlement_to_id
            DefaultCurrencyCode_,               //  currency
            DefaultTimeZoneCode_,               //  time_zone_from
            "7",                                //  t_type_id
            ToString(DefaultSupplierId_),       //  supplier_id
            "10:20:15",                         //  time_from
        };
    }
    TThreadTariff CreateModel() {
        TThreadTariff s;
        s.SetId(1);
        s.SetThreadId(DefaultThreadId_);
        s.SetFromStationId(DefaultFromStationId_);
        s.SetToStationId(DefaultToStationId_);
        s.SetTariff(222.0);
        auto& yearsDays = *s.MutableYearDaysFrom();
        for (int i = 0; i < 12; ++i) {
            yearsDays.Add(2147483647);
        }
        s.SetFromSettlementId(DefaultFromSettlementId_);
        s.SetToSettlementId(DefaultToSettlementId_);
        s.SetCurrencyId(DefaultCurrencyId_);
        s.SetFromTimeZoneId(DefaultTimeZoneId_);
        s.SetTransportType(TTransport::TYPE_HELICOPTER);
        s.SetSupplierId(DefaultSupplierId_);
        s.SetFromTime((10 * 60 + 20) * 60 + 15);

        return s;
    }
    void TestEmpty();
    void TestOneRecord();
};

UNIT_TEST_SUITE_REGISTRATION(TThreadTariffDumperTest);

void TThreadTariffDumperTest::TestEmpty() {
    TMap<TString, i32> threadUidToId;
    TSet<i32> stationIds;
    TSet<i32> settlementIds;
    TSet<i32> supplierIds;
    TMap<TString, i32> currencyCodeToId;
    Fetcher_.Add({});

    UNIT_ASSERT_EXCEPTION(
        Dumper_->Dump(threadUidToId, stationIds, settlementIds, supplierIds, currencyCodeToId),
        yexception);
}

void TThreadTariffDumperTest::TestOneRecord() {
    Fetcher_.Add({CreateRow()});

    Dumper_->Dump(ThreadUidToId_, StationIds_, SettlementIds_, SupplierIds_, CurrencyCodeToId_);
    UNIT_ASSERT_STRINGS_EQUAL(
        ToJson(ObjectWriter_.GetResult()),
        ToJson(TVector<TThreadTariff>{CreateModel()}));
}
