#include "railway_timezone_provider.h"

using namespace NRasp::NData;

namespace NRasp {
    namespace NDumper {
        TRailwayTimezoneProvider::TRailwayTimezoneProvider(
            const TVector<ExpressToCountryShort>& expressToCountryShort,
            const THashMap<i32, THashMap<i32, TString>>& stationIdToCodes,
            const THashMap<i32, i32>& countryIdToCapitalTimeZoneId,
            const TTimeZoneProvider& timeZoneProvider)
            : StationIdToCodes_(stationIdToCodes)
            , CountryIdToCapitalTimeZoneId_(countryIdToCapitalTimeZoneId)
            , TimeZoneProvider_(timeZoneProvider)
        {
            for (auto i : expressToCountryShort) {
                auto regexp = std::regex(i.CodeRe.c_str());
                RegexExpressToCountry_.emplace_back(regexp, i);
            }
        }

        i32 TRailwayTimezoneProvider::GetTimezoneIdForStation(i32 stationId) const {
            auto defaultTimeZoneId = TimeZoneProvider_.GetTimeZoneId("");
            auto stationCodes = StationIdToCodes_.find(stationId);

            if (stationCodes == StationIdToCodes_.end()) {
                    INFO_LOG << "Cannot find any codes for station id: " << stationId << Endl;
                    return defaultTimeZoneId;
            }

            auto expressCode = stationCodes->second.find(CODE_SYSTEM_EXPRESS);
            if (expressCode == stationCodes->second.end()){
                INFO_LOG << "Cannot find express code for station id: " << stationId << Endl;
                return defaultTimeZoneId;
            }


            auto expressToCountry = FindExpressToCountry(expressCode->second);
            if (expressToCountry.Empty()) {
                INFO_LOG << "Cannot find value for express code: " << expressCode->second << Endl;
                return defaultTimeZoneId;
            }

            if (!expressToCountry->TimeZone.empty()) {
                auto timeZoneId = TimeZoneProvider_.GetTimeZoneId(expressToCountry->TimeZone);
                return timeZoneId;
            }

            auto countryIdToCapitalTimeZoneId = CountryIdToCapitalTimeZoneId_.find(expressToCountry->CountryId);
            if (countryIdToCapitalTimeZoneId == CountryIdToCapitalTimeZoneId_.end()) {
                INFO_LOG << "Cannot find capital time zone for countryId: " << expressToCountry->CountryId << Endl;
                return defaultTimeZoneId;
            }

            return countryIdToCapitalTimeZoneId->second;
        }

        TMaybe<ExpressToCountryShort> TRailwayTimezoneProvider::FindExpressToCountry(const TString& expressCode) const {
            auto code = expressCode.c_str();
            for (auto regexExpressToCountry : RegexExpressToCountry_) {
                if (std::regex_match(code, regexExpressToCountry.Regex)) {
                    return regexExpressToCountry.ExpressToCountry;
                }
            }

            return Nothing();
        }

        RegexExpressToCountryShort::RegexExpressToCountryShort(std::regex regex, ExpressToCountryShort expressToCountry) {
            Regex = regex;
            ExpressToCountry = expressToCountry;
        }
    }
}
