#include "railway_timezone_provider.h"

#include <travel/rasp/rasp_data/dumper/lib/tester/fabrics.h>

#include <library/cpp/testing/unittest/registar.h>
#include <util/string/cast.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

class TRailwayTimezoneProviderTest: public TTestBase {
private:
    UNIT_TEST_SUITE(TRailwayTimezoneProviderTest);
    UNIT_TEST(TestRailwayTimezone);
    UNIT_TEST(TestSpecificTimezone);
    UNIT_TEST(TestUnknownStationTimezone);
    UNIT_TEST(TestUnknownExpressTimezone);
    UNIT_TEST_SUITE_END();

private:
    THashMap<i32, THashMap<i32, TString>> StationIdToExpressCode_;
    THashMap<i32, i32> CountryIdToCapitalTimeZoneId_;
    TVector<ExpressToCountryShort> ExpressToCountries_;
    THolder<TTimeZoneProvider> TzProvider_;
    THolder<TRailwayTimezoneProvider> RailwayTimezoneProvider_;

    const i32 DefaultTimezoneId = 1;
    const i32 CapitalTimezoneId = 2;
    const i32 SpecificTimezoneId = 3;
    const i32 UsualStationId = 444;
    const i32 SpecificStationId = 555;
    const i32 UnknownExpressStationId = 666;
    const i32 CountryId = 123;

public:
    void SetUp() override {
        ExpressToCountries_ = {ExpressToCountryShort("10.*", CountryId, ""), ExpressToCountryShort("11.*", CountryId, "Asia/Yekaterinburg")};
        StationIdToExpressCode_ = {{UsualStationId, {{CODE_SYSTEM_EXPRESS, "1001"}}},
                                   {SpecificStationId,{{CODE_SYSTEM_EXPRESS, "1101"}}},
                                   {UnknownExpressStationId, {{CODE_SYSTEM_EXPRESS,"2000"}}}};
        CountryIdToCapitalTimeZoneId_ = {{CountryId, CapitalTimezoneId}};
        TzProvider_ = MakeHolder<TTimeZoneProvider>(
            TVector<TTimeZone>{CreateTimeZone(DefaultTimezoneId, "UTC"), CreateTimeZone(CapitalTimezoneId, "Europe/Moscow"), CreateTimeZone(SpecificTimezoneId, "Asia/Yekaterinburg")},
            "UTC");
        RailwayTimezoneProvider_ = MakeHolder<TRailwayTimezoneProvider>(
            ExpressToCountries_,
            StationIdToExpressCode_,
            CountryIdToCapitalTimeZoneId_,
            *TzProvider_);
    }

    void TestRailwayTimezone();
    void TestSpecificTimezone();
    void TestUnknownStationTimezone();
    void TestUnknownExpressTimezone();
};

UNIT_TEST_SUITE_REGISTRATION(TRailwayTimezoneProviderTest);

void TRailwayTimezoneProviderTest::TestRailwayTimezone() {
    auto result = RailwayTimezoneProvider_->GetTimezoneIdForStation(UsualStationId);
    UNIT_ASSERT_EQUAL(result, CapitalTimezoneId);
}

void TRailwayTimezoneProviderTest::TestSpecificTimezone() {
    auto result = RailwayTimezoneProvider_->GetTimezoneIdForStation(SpecificStationId);
    UNIT_ASSERT_EQUAL(result, SpecificTimezoneId);
}

void TRailwayTimezoneProviderTest::TestUnknownStationTimezone() {
    auto result = RailwayTimezoneProvider_->GetTimezoneIdForStation(0);
    UNIT_ASSERT_EQUAL(result, DefaultTimezoneId);
}

void TRailwayTimezoneProviderTest::TestUnknownExpressTimezone() {
    auto result = RailwayTimezoneProvider_->GetTimezoneIdForStation(UnknownExpressStationId);
    UNIT_ASSERT_EQUAL(result, DefaultTimezoneId);
}
