#include "timezone_loader.h"

#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/list.h>
#include <util/generic/vector.h>
#include <util/generic/ptr.h>
#include <util/stream/str.h>
#include <travel/proto/dicts/rasp/timezone.pb.h>
#include <travel/rasp/rasp_data/dumper/lib/fetcher/fake_fetcher.h>
#include <travel/rasp/rasp_data/dumper/lib/object_writer/fake_object_writer.h>
#include <travel/rasp/rasp_data/dumper/lib/tester/json.h>
#include <google/protobuf/util/json_util.h>
#include <google/protobuf/util/message_differencer.h>
#include <google/protobuf/stubs/status.h>
#include <google/protobuf/io/zero_copy_stream_impl_lite.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

class TTimeZoneLoaderTest: public TTestBase {
private:
    UNIT_TEST_SUITE(TTimeZoneLoaderTest);
    UNIT_TEST(TestNoTimeZones);
    UNIT_TEST(TestAllHaveSomeTimeZone);
    UNIT_TEST(TestOnlySettlementTableHasTimeZones);
    UNIT_TEST(TestOnlyStationTableHasTimeZones);
    UNIT_TEST(TestOnlyRTStationTableHasTimeZones);
    UNIT_TEST(TestUnknownTimeZones);
    UNIT_TEST_SUITE_END();

    TTimeZone CreateTTimeZone(i32 id, TString code) {
        TTimeZone t;
        t.SetId(id);
        t.SetCode(code);
        return t;
    }

private:
    TFakeFetcher Fetcher_;
    TFakeObjectWriter<TTimeZone> ObjectWriter_;
    THolder<TTimeZoneLoader> Loader_;

    const TString MskTz_ = "Europe/Moscow";
    const TString EkbTz_ = "Asia/Yekaterinburg";
    const TString VldTz_ = "Asia/Vladivostok";

public:
    void SetUp() override {
        Fetcher_ = {};
        ObjectWriter_ = TFakeObjectWriter<TTimeZone>();
        Loader_ = MakeHolder<TTimeZoneLoader>(Fetcher_, ObjectWriter_);
    }
    void TestNoTimeZones();
    void TestAllHaveSomeTimeZone();
    void TestOnlySettlementTableHasTimeZones();
    void TestOnlyStationTableHasTimeZones();
    void TestOnlyRTStationTableHasTimeZones();
    void TestUnknownTimeZones();
};

UNIT_TEST_SUITE_REGISTRATION(TTimeZoneLoaderTest);

void TTimeZoneLoaderTest::TestNoTimeZones() {
    Fetcher_.Add({}); // www_settlements
    Fetcher_.Add({}); // www_station
    Fetcher_.Add({}); // www_rtstation
    UNIT_ASSERT_EXCEPTION(Loader_->Load().size(), yexception);
}

void TTimeZoneLoaderTest::TestAllHaveSomeTimeZone() {
    Fetcher_.Add({{MskTz_}}); // www_settlements
    Fetcher_.Add({{VldTz_}}); // www_station
    Fetcher_.Add({{EkbTz_}}); // www_rtstation
    auto result = Loader_->Load();
    auto expected = TVector<TTimeZone>{
        CreateTTimeZone(0, MskTz_),
        CreateTTimeZone(1, VldTz_),
        CreateTTimeZone(2, EkbTz_)};

    UNIT_ASSERT_STRINGS_EQUAL(ToJson(result), ToJson(expected));
}

void TTimeZoneLoaderTest::TestOnlySettlementTableHasTimeZones() {
    Fetcher_.Add({{MskTz_}, {VldTz_}}); // www_settlements
    Fetcher_.Add({});                   // www_station
    Fetcher_.Add({});                   // www_rtstation
    auto result = Loader_->Load();
    auto expected = TVector<TTimeZone>{
        CreateTTimeZone(0, MskTz_),
        CreateTTimeZone(1, VldTz_),
    };

    UNIT_ASSERT_STRINGS_EQUAL(ToJson(result), ToJson(expected));
}

void TTimeZoneLoaderTest::TestOnlyStationTableHasTimeZones() {
    Fetcher_.Add({});                   // www_setlement
    Fetcher_.Add({{MskTz_}, {VldTz_}}); // www_station
    Fetcher_.Add({});                   // www_rtstation
    auto result = Loader_->Load();
    auto expected = TVector<TTimeZone>{
        CreateTTimeZone(0, MskTz_),
        CreateTTimeZone(1, VldTz_),
    };

    UNIT_ASSERT_STRINGS_EQUAL(ToJson(result), ToJson(expected));
}

void TTimeZoneLoaderTest::TestOnlyRTStationTableHasTimeZones() {
    Fetcher_.Add({});                   // www_setlement
    Fetcher_.Add({});                   // www_station
    Fetcher_.Add({{MskTz_}, {VldTz_}}); // www_rtstation
    auto result = Loader_->Load();
    auto expected = TVector<TTimeZone>{
        CreateTTimeZone(0, MskTz_),
        CreateTTimeZone(1, VldTz_),
    };

    UNIT_ASSERT_STRINGS_EQUAL(ToJson(result), ToJson(expected));
}

void TTimeZoneLoaderTest::TestUnknownTimeZones() {
    Fetcher_.Add({{"UNKOWN1"}}); // www_setlement
    Fetcher_.Add({{"UNKOWN2"}}); // www_station
    Fetcher_.Add({{"UNKOWN3"}}); // www_rtstation

    UNIT_ASSERT_EXCEPTION(Loader_->Load(), yexception);
}
