#include "timezone_provider.h"

#include <library/cpp/testing/unittest/registar.h>
#include <util/generic/list.h>
#include <util/generic/vector.h>
#include <travel/proto/dicts/rasp/timezone.pb.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

class TTimeZoneProviderTest: public TTestBase {
private:
    UNIT_TEST_SUITE(TTimeZoneProviderTest);
    UNIT_TEST_EXCEPTION(TestUnkownDefaultTimezone, TUnknownDefaultTimeZone);
    UNIT_TEST_EXCEPTION(TestDefaultTimezoneExistsButDontPresent, TUnpresentDefaultTimeZone);
    UNIT_TEST(TestOnlyDefaultTimeZone);
    UNIT_TEST(TestDefaultAndAnotherOneTimeZone);
    UNIT_TEST_SUITE_END();

    TVector<TTimeZone> TimeZones_;
    TString DefaultTimeZoneCode_;

public:
    void SetUp() override {
        DefaultTimeZoneCode_ = "Europe/Moscow";
        TimeZones_ = {};
    }
    void AddTimeZone(i32 id, TString code) {
        TTimeZone t;
        t.SetId(id);
        t.SetCode(code);
        TimeZones_.emplace_back(t);
    }
    void TestUnkownDefaultTimezone();
    void TestOnlyDefaultTimeZone();
    void TestDefaultTimezoneExistsButDontPresent();
    void TestDefaultAndAnotherOneTimeZone();
};

UNIT_TEST_SUITE_REGISTRATION(TTimeZoneProviderTest);

void TTimeZoneProviderTest::TestUnkownDefaultTimezone() {
    TTimeZoneProvider provider(TimeZones_, "");
}

void TTimeZoneProviderTest::TestDefaultTimezoneExistsButDontPresent() {
    using namespace NRasp;
    AddTimeZone(1, "Asia/Vladivostok");

    TTimeZoneProvider provider(TimeZones_, DefaultTimeZoneCode_);
}

void TTimeZoneProviderTest::TestOnlyDefaultTimeZone() {
    using namespace NRasp;
    AddTimeZone(1, DefaultTimeZoneCode_);

    TTimeZoneProvider provider(TimeZones_, DefaultTimeZoneCode_);

    UNIT_ASSERT_EQUAL(provider.GetTimeZoneId(DefaultTimeZoneCode_), 1);
    UNIT_ASSERT_EQUAL(provider.GetTimeZoneId("XXX"), 1);

    UNIT_ASSERT_EQUAL(TString(provider.GetTimeZone(1).name()), DefaultTimeZoneCode_);
    UNIT_ASSERT_EQUAL(TString(provider.GetTimeZone(2).name()), DefaultTimeZoneCode_);
}

void TTimeZoneProviderTest::TestDefaultAndAnotherOneTimeZone() {
    using namespace NRasp;
    TString vldTimeZoneCode = "Asia/Vladivostok";
    AddTimeZone(1, DefaultTimeZoneCode_);
    AddTimeZone(2, vldTimeZoneCode);

    TTimeZoneProvider provider(TimeZones_, DefaultTimeZoneCode_);

    UNIT_ASSERT_EQUAL(provider.GetTimeZoneId(DefaultTimeZoneCode_), 1);
    UNIT_ASSERT_EQUAL(provider.GetTimeZoneId(vldTimeZoneCode), 2);
    UNIT_ASSERT_EQUAL(provider.GetTimeZoneId("XXX"), 1);

    UNIT_ASSERT_EQUAL(TString(provider.GetTimeZone(1).name()), DefaultTimeZoneCode_);
    UNIT_ASSERT_EQUAL(TString(provider.GetTimeZone(2).name()), vldTimeZoneCode);
    UNIT_ASSERT_EQUAL(TString(provider.GetTimeZone(1).name()), DefaultTimeZoneCode_);
}
