#include "title_common.h"

#include <travel/rasp/rasp_data/dumper/lib/tester/json.h>

#include <library/cpp/testing/unittest/registar.h>
#include <library/cpp/json/json_reader.h>

using namespace NRasp::NData;
using namespace NRasp::NDumper;

Y_UNIT_TEST_SUITE(TParseTitleCommon) {
    Y_UNIT_TEST(InvalidJson) {
        i32 titleId = 999;
        TSet<i32> stationIds;
        TSet<i32> settlementIds;
        TString rawTitle = "}{";
        TThreadTitle title;

        UNIT_CHECK_GENERATED_EXCEPTION(
            ParseTitleCommon(titleId, rawTitle, stationIds, settlementIds, &title),
            NJson::TJsonException);
    }

    Y_UNIT_TEST(CombinedThread) {
        i32 titleId = 999;
        TSet<i32> stationIds;
        TSet<i32> settlementIds;
        TString rawTitle = "{\"is_combined\": true, \"title_parts\": []}";
        TThreadTitle title;
        UNIT_ASSERT_EQUAL(ParseTitleCommon(titleId, rawTitle, stationIds, settlementIds, &title), true);

        TThreadTitle expectedTitle;
        expectedTitle.SetIsCombined(true);

        UNIT_ASSERT_STRINGS_EQUAL(MessageToJson(title), MessageToJson(expectedTitle));
    }

    Y_UNIT_TEST(ThreadWithRing) {
        i32 titleId = 999;
        TSet<i32> stationIds;
        TSet<i32> settlementIds;
        TVector<TString> rawTitles{
            "{\"add_ring\": true, \"title_parts\": []}",
            "{\"add_circular\": true, \"title_parts\": []}",
            "{\"add_circular_mta\": true, \"title_parts\": []}",
        };
        TThreadTitle expectedTitle;
        expectedTitle.SetIsRing(true);

        for (const auto& rawTitle : rawTitles) {
            TThreadTitle title;
            UNIT_ASSERT_EQUAL(ParseTitleCommon(titleId, rawTitle, stationIds, settlementIds, &title), true);

            UNIT_ASSERT_STRINGS_EQUAL(MessageToJson(title), MessageToJson(expectedTitle));
        }
    }

    Y_UNIT_TEST(ThreadWithoutRing) {
        i32 titleId = 999;
        TSet<i32> stationIds;
        TSet<i32> settlementIds;

        TString rawTitle = "{\"title_parts\": []}";
        TThreadTitle expectedTitle;

        TThreadTitle title;
        UNIT_ASSERT_EQUAL(ParseTitleCommon(titleId, rawTitle, stationIds, settlementIds, &title), true);

        UNIT_ASSERT_STRINGS_EQUAL(MessageToJson(title), MessageToJson(expectedTitle));
    }
    Y_UNIT_TEST(CheckOkParts) {
        i32 titleId = 999;
        TSet<i32> stationIds{2, 100};
        TSet<i32> settlementIds{213, 10};
        TString rawTitle = "{\"title_parts\": [\"c213\", \"s2\", {\"settlement\":\"c10\", \"station\": \"s100\"}]}";
        TThreadTitle expectedTitle;
        {
            auto& part = *expectedTitle.AddTitleParts();
            part.SetSettlementId(213);
        }
        {
            auto& part = *expectedTitle.AddTitleParts();
            part.SetStationId(2);
        }
        {
            auto& part = *expectedTitle.AddTitleParts();
            part.SetSettlementId(10);
            part.SetStationId(100);
        }
        TThreadTitle title;
        UNIT_ASSERT_EQUAL(ParseTitleCommon(titleId, rawTitle, stationIds, settlementIds, &title), true);
        UNIT_ASSERT_STRINGS_EQUAL(MessageToJson(title), MessageToJson(expectedTitle));
    }

    Y_UNIT_TEST(BrokenTitleParts) {
        i32 titleId = 999;
        TSet<i32> stationIds;
        TSet<i32> settlementIds;
        {
            TString rawTitle = "{\"title_parts\": [true]}";
            TThreadTitle title;
            UNIT_ASSERT_EQUAL(ParseTitleCommon(titleId, rawTitle, stationIds, settlementIds, &title), false);
        }
        {
            TString rawTitle = "{\"title_parts\": [1]}";
            TThreadTitle title;
            UNIT_ASSERT_EQUAL(ParseTitleCommon(titleId, rawTitle, stationIds, settlementIds, &title), false);
        }
        {
            TString rawTitle = "{\"title_parts\": [null]}";
            TThreadTitle title;
            UNIT_ASSERT_EQUAL(ParseTitleCommon(titleId, rawTitle, stationIds, settlementIds, &title), false);
        }
        {
            TString rawTitle = "{\"title_parts\": [\"g123\"]}";
            TThreadTitle title;
            UNIT_ASSERT_EQUAL(ParseTitleCommon(titleId, rawTitle, stationIds, settlementIds, &title), false);
        }
        {
            TString rawTitle = "{\"title_parts\": [{\"settlement\": \"s123\", \"station\": \"s10\"}]}";
            TThreadTitle title;
            UNIT_ASSERT_EQUAL(ParseTitleCommon(titleId, rawTitle, stationIds, settlementIds, &title), false);
        }
        {
            TString rawTitle = "{\"title_parts\": [{\"settlement\": \"c123\", \"station\": \"c10\"}]}";
            TThreadTitle title;
            UNIT_ASSERT_EQUAL(ParseTitleCommon(titleId, rawTitle, stationIds, settlementIds, &title), false);
        }
    }
}
