# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

from logging import Logger
from collections import defaultdict

from travel.rasp.rasp_data.resource_checker.data_provider import DataProvider
from travel.rasp.rasp_data.resource_checker.check_exception import CheckException

RUSSIA_ID = 225
MOSCOW_ID = 213
CAPITAL_MAJORITY = 1


def check_capitals(data_provider, log):
    # type: (DataProvider, Logger) -> None

    capitals = [
        settlement for settlement in data_provider.settlement_repo.itervalues()
        if settlement.Majority == CAPITAL_MAJORITY
    ]
    capitals_by_countries = defaultdict(list)
    for capital in capitals:
        if capital.CountryId == 0:
            raise CheckException('Capital without country: {}'.format(capital.Id))
        country = data_provider.country_repo.get(capital.CountryId)
        if country is None:
            raise CheckException(
                "Can't find country with id `{}` for capital `{}`".format(capital.CountryId, capital.Id)
            )
        capitals_by_countries[country.Id].append(capital.Id)

    countries_with_multiple_capitals = [
        (country_id, capital_ids) for country_id, capital_ids in capitals_by_countries.items()
        if len(capital_ids) != 1
    ]
    if len(countries_with_multiple_capitals):
        raise CheckException(
            'There are countries with multiple capitals: {}'.format(
                ', '.join(
                    '({}: {})'.format(country_id, capital_ids)
                    for (country_id, capital_ids) in countries_with_multiple_capitals
                )
            )
        )

    log.info('All capitals are OK')


def check_moscow_is_a_capital_of_russia(data_provider, log):
    # type: (DataProvider, Logger) -> None

    russia_capitals = [
        settlement for settlement in data_provider.settlement_repo.itervalues()
        if settlement.CountryId == RUSSIA_ID and settlement.Majority == CAPITAL_MAJORITY
    ]
    if len(russia_capitals) == 0:
        raise CheckException('Russia has no capitals')

    if len(russia_capitals) > 1:
        raise CheckException(
            'Russia has multiple capitals: {}'.format(
                ', '.join(str(capital.Id) for capital in russia_capitals)
            )
        )

    maybe_moscow = russia_capitals[0]
    if maybe_moscow.Id != MOSCOW_ID:
        raise CheckException('Wrong capital of Russia: {}'.format(maybe_moscow.Id))

    log.info('Moscow is a capital of Russia')


CHECKS = [
    check_capitals,
    check_moscow_is_a_capital_of_russia
]
