# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import codecs
import os
import cPickle as pickle
import subprocess

import numpy as np
from django.conf import settings
from pathlib2 import Path

from common.db.switcher import switcher
from common.dynamic_settings.core import DynamicSetting
from common.dynamic_settings.default import conf
from travel.rasp.library.python.common23.logging import create_current_file_run_log
from mapping.generators.utils import simplify_and_encode_polyline


conf.register_settings(
    ENABLE_GEN_RAILWAY_DATA=DynamicSetting(
        True,
        cache_time=10 * 60,
        description='Включает генерацию жд геометрии в переключении')
)

log = logging.getLogger(__name__)


def log_railway_warnings(railway_dir):
    log_file = 'badStationLocation.txt'

    f = codecs.open(log_file, 'r', encoding='utf-8')

    critical = []
    warnings = []

    for line in f:
        try:
            _, id, title, uid = line.rstrip('\n').split('\t')
        except:
            continue

        if '!!! empty_loc' in line:
            critical.append((id, title, uid))
        elif 'empty_loc' in line:
            warnings.append((id, title, uid))

    message = []

    if critical:
        message.append(u"Важные:")

        message.extend("%s %s %s" % s for s in critical)

    if warnings:
        if critical:
            message.append('')

        message.append(u'Остальные:')

        message.extend("%s %s %s" % s for s in warnings)

    if message:
        subject = u'Станции без координат'

        message = '\n'.join(message)

        log.warning('{}. {}'.format(subject, message))

    f.close()


def run_railway(railway_dir, pathfinder_path, mif_path, log):
    cwd = os.getcwd()

    os.chdir(railway_dir)

    log.info("Generating railway data")

    try:
        process = subprocess.Popen(
            "./railway {} {}".format(pathfinder_path, mif_path),
            shell=True, stderr=subprocess.STDOUT, stdout=subprocess.PIPE
        )

        for line in process.stdout:
            log.info(line.rstrip().decode('utf-8'))

        if process.wait() != 0:
            raise Exception("railway blackbox failed, look into log")

        log_railway_warnings(railway_dir)

        try:
            subprocess.check_call("sh make_station_report.sh",
                                  shell=True, stderr=subprocess.STDOUT, stdout=subprocess.PIPE)
            log_path = settings.LOG_PATH + "/"
            subprocess.check_call("cp railway_interpolated_stations.txt railway_far_stations.txt %s" % log_path,
                                  shell=True, stderr=subprocess.STDOUT, stdout=subprocess.PIPE)
        except:
            log.info("Warning! Make station report failed!")
    finally:
        os.chdir(cwd)


def run(railway_path, pathfinder_data_path, mif_path):
    create_current_file_run_log()

    dir_path_db = os.path.join(pathfinder_data_path, switcher.get_db_alias())
    run_railway(railway_path, dir_path_db, mif_path, log)

    map_path = os.path.join(railway_path, 'edge2curve.txt')

    segments = {}
    stations = {}

    log.info('Processing railway data')

    data = open(map_path, 'r')

    for line in data:
        st1, st2, numbers, points = line.split(None, 3)

        st1, st2 = int(st1), int(st2)
        numbers = [None] if numbers == 'none' else numbers.decode('utf-8').split(',')

        # Катин порядок не совпадает с яндекс.картами, делаем fliplr
        points = np.fliplr(np.fromstring(points, sep=" ").reshape([-1, 2]))

        polyline, levels = simplify_and_encode_polyline(points)

        for number in numbers:
            segments[(st1, st2, number)] = polyline, levels

        stations[st1] = list(points[0])
        stations[st2] = list(points[-1])

    data.close()

    railways_pickle_filepath = Path(settings.GEOMETRY_BASE_PATH, switcher.get_db_alias(), 'railways.pickle')
    railways_pickle_filepath.parent.mkdir(parents=True, exist_ok=True)
    with railways_pickle_filepath.open('wb') as out:
        pickle.dump({
            'segments': segments,
            'stations': stations,
        }, out)

    log.info(u'Done')


if __name__ == '__main__':
    run('', '', '',)
