# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import gzip
import os
import shutil
from xml.etree import ElementTree

from travel.rasp.library.python.api_clients.market_cms import MarketCmsClient
from common.db.mds.clients import mds_s3_public_client


log = logging.getLogger(__name__)

TRAVEL_PORTAL_URL = 'https://travel.yandex.ru'
MARKET_CMS_DEFAULT_URL = 'http://templator.vs.market.yandex.net:29338'
MDS_JOURNAL_SITEMAP_PREFIX = 'journal/sitemaps'


def generate_urls(client):
    pages_data = client.get_context_page()
    pages_urls = []
    for p in pages_data:
        semantic_prefix = '/'
        if 'semanticPrefix' in p:
            semantic_prefix = p['semanticPrefix']
        pages_urls.append('{}{}{}/'.format(TRAVEL_PORTAL_URL, semantic_prefix, p['semanticId']))
    return pages_urls


def _recreate_local_dir(path):
    if os.path.exists(path):
        shutil.rmtree(path)
    os.mkdir(path)


def url2xml(url, is_root_file):
    url_tag = ElementTree.Element('sitemap' if is_root_file else 'url')
    loc_tag = ElementTree.Element('loc')
    loc_tag.text = url
    url_tag.append(loc_tag)
    return url_tag


def generate_sitemap_file(file_path, urls, is_root_file):
    # https://www.sitemaps.org/protocol.html
    root = ElementTree.Element(
        'sitemapindex' if is_root_file else 'urlset',
        attrib={
            'xmlns': 'http://www.sitemaps.org/schemas/sitemap/0.9'
        }
    )
    for url in urls:
        root.append(url2xml(url, is_root_file))
    tree = ElementTree.ElementTree(root)

    with open(file_path, 'wb') as fgz:
        with gzip.GzipFile(fileobj=fgz, mode="w") as f:
            tree.write(f, encoding='utf-8', xml_declaration=True)


def run(market_cms_host=MARKET_CMS_DEFAULT_URL):
    log.info('generate_journal_sitemap: run_script')

    log.info('generate_journal_sitemap: loading data from {}'.format(market_cms_host))
    client = MarketCmsClient(market_cms_host)
    pages_urls = generate_urls(client)

    sitemaps_local_root = 'sitemaps'
    _recreate_local_dir(sitemaps_local_root)
    sitemap_filename = 'sitemap.xml.gz'
    sitemap_local_path = os.path.join(sitemaps_local_root, sitemap_filename)

    log.info('generate_journal_sitemap: generating file {}/{}'.format(sitemaps_local_root, sitemap_filename))
    generate_sitemap_file(sitemap_local_path, pages_urls, is_root_file=False)

    mds_key = '{}/{}'.format(MDS_JOURNAL_SITEMAP_PREFIX, sitemap_filename)
    log.info('Uploading {} -> {}'.format(sitemap_local_path, mds_key))
    mds_s3_public_client.save_data(key=mds_key, data=open(sitemap_local_path, 'rb'))

    log.info('generate_journal_sitemap: run_script end')
