# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import json
import logging
import pickle

import numpy as np
from pathlib2 import Path

from django.conf import settings

from common.db.switcher import switcher
from common.models.geo import Station
from common.utils.geo import GeoPoint
from common.utils.progress import PercentageStatus
from travel.rasp.library.python.common23.logging import create_current_file_run_log
from mapping.generators.utils import simplify_and_encode_polyline
from common.models_admin.geo import RoutePath


log = logging.getLogger(__name__)


def prepare_curve(points):
    points_array = np.array(points)

    return simplify_and_encode_polyline(points_array)


def check_direction_is_correct(arc, station_from, station_to):
    if station_from and station_to:
        start, end = arc[0], arc[-1]
        start_point = GeoPoint(start[1], start[0])
        end_point = GeoPoint(end[1], end[0])

        start_to_from = start_point.distance(station_from)
        start_to_to = start_point.distance(station_to)
        end_to_from = end_point.distance(station_from)
        end_to_to = end_point.distance(station_to)

        if start_to_from > start_to_to and end_to_to > end_to_from:
            return False

    return True


def run():
    create_current_file_run_log()

    segments = {}

    total = RoutePath.objects.all().count()

    status = PercentageStatus(total, log)

    route_paths = list(RoutePath.objects.all())

    stations_ids = set()

    for route_path in route_paths:
        stations_ids.add(route_path.station_from_id)
        stations_ids.add(route_path.station_to_id)

    stations = Station.objects.in_bulk(stations_ids)

    for route_path in route_paths:
        direct_points = back_points = None

        if route_path.data_direct and route_path.status_direct == RoutePath.STATUS_CONFIRMED:
            direct_points = json.loads(route_path.data_direct)

            station_from = stations.get(route_path.station_from_id)
            station_to = stations.get(route_path.station_to_id)

            if not check_direction_is_correct(direct_points, station_from, station_to):
                direct_points.reverse()

        # TODO: нужно сначала понять, как в реальности будут использованы обратные нитки
        # if route_path.data_back and route_path.status_back == RoutePath.STATUS_CONFIRMED:
        #     back_points = json.loads(route_path.data_back)
        #
        #     if not check_direction_is_correct(back_points, station_to, station_from):
        #         back_points.reverse()
        #
        #     back = prepare_curve(back_points)

        if route_path.for_two_directions:
            if direct_points and not back_points:
                back_points = direct_points[::-1]
            elif back_points and not direct_points:
                direct_points = back_points[::-1]

        key = (route_path.station_from_id, route_path.station_to_id)

        segments[key] = (
            prepare_curve(direct_points) if direct_points else None,
            prepare_curve(back_points) if back_points else None,
            # Теперь мы всегда указываем кривые для обоих направлений, признак оставлен для обратной соdместимости
            False
        )

        status.step()

    limepaths_pickle_path = Path(settings.GEOMETRY_BASE_PATH, switcher.get_db_alias(), 'limepaths.pickle')
    limepaths_pickle_path.parent.mkdir(parents=True, exist_ok=True)
    with limepaths_pickle_path.open('wb') as out:
        pickle.dump({
            'segments': segments,
        }, out)

    log.info(u'Done')


if __name__ == '__main__':
    run()
