# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
import sqlite3

from importlib import import_module

from django.conf import settings

# from travel.rasp.library.python.common23.logging import add_stdout_handler
from travel.rasp.rasp_scripts.scripts.long_haul.export import generators, validation

log = logging.getLogger(__name__)

# список версий, которые мы поддерживаем
# https://wiki.yandex-team.ru/maps/dev/core/masstransit/dataformat/
SUPPORTED_VERSIONS = [2.5, 2.6]

GENERATORS = (
    ('routes', generators.RoutesGenerator),
    ('threads', generators.ThreadsGenerator),
    ('thread_stops', generators.ThreadStopsGenerator),
    ('stops', generators.StopsGenerator),
    ('geometry', generators.GeometryGenerator),
    ('frequency', None),
    ('timetable', generators.TimetableGenerator),
    ('calendar', generators.CalendarGenerator),
    ('transitions', None),
    ('l10n', generators.L10nGenerator),
)

EXPORTS = {
    (2.5,): {'only_urban': False},
    (2.6,): {'only_urban': False}
}

# internal
VERSIONS = {}

for version in SUPPORTED_VERSIONS:
    sanitized = 'v%s' % str(version).replace('.', '_')
    module = import_module('travel.rasp.rasp_scripts.scripts.long_haul.export.%s' % sanitized)
    VERSIONS[version] = module


class MultiExporter:
    def __init__(self, versions=SUPPORTED_VERSIONS, log=log, skip_generators=None):
        self.versions = versions
        self.log = log
        self.outputs = {}
        self.formatters = {}
        self.skip = skip_generators or []
        self.validation_conn = None

        for version in versions:
            if version not in VERSIONS:
                raise ValueError(
                    'Version %s is not supported. Supported versions are: %s' %
                    (version, ", ".join(map(str, SUPPORTED_VERSIONS)))
                )

            self.outputs[version] = {}
            self.formatters[version] = {}

            for export, generator in GENERATORS:
                if export in self.skip:
                    continue

                self.formatters[version][export] = getattr(VERSIONS[version], export, None)()

                # touch file
                self.output_for(version, export)

    def export(self):
        try:
            for versions, kwargs in EXPORTS.items():
                self.validation_conn = sqlite3.connect(validation.DB_FILE)
                validation.init_db(self.validation_conn)
                _versions = [v for v in self.versions if v in versions]

                if not _versions:
                    continue

                self.log.info(u'Начинаем экспорт гортранса, версии %s' % ", ".join(map(str, _versions)))

                # последовательно дергаем каждую ручку, записывая одинаковые данные в разных форматах в разные версии
                for export, generator in GENERATORS:
                    if export in self.skip:
                        continue

                    if generator is None:
                        continue

                    self.log.info(u'Экспортируем %s' % export)

                    for obj in generator(**kwargs).generate(self.log):
                        for version in _versions:
                            formatter = self.formatters[version][export]

                            formatter.write(obj, self.output_for(version, export))

                        generator.insert_into_validation_table(obj, self.validation_conn)

                self.validate_export()
        finally:
            self.close_outputs()

    def validate_export(self):
        self.log.info(u'Начинаем проверку')
        validation.process(self.log, self.validation_conn)
        validation.remove_db()

    def output_for(self, version, export):
        if export in self.outputs[version]:
            return self.outputs[version][export]
        else:
            self.outputs[version][export] = open(self.output_path_for(version, export), 'wb')

            return self.outputs[version][export]

    def close_outputs(self):
        for exports in self.outputs.values():
            for output in exports.values():
                output.close()

    def output_path_for(self, version, export):
        return os.path.join(self.directory(version), export)

    def directory(self, version):
        dir_name = os.path.join(settings.EXPORT_URBAN_PUBLIC_PATH, str(version))

        if not os.path.exists(dir_name):
            os.makedirs(dir_name)

        return dir_name


def export(log=log, verbose=False, versions=SUPPORTED_VERSIONS, skip_generators=None):
    log.info(u'Экспорт дальних рейсов')

    if skip_generators:
        log.info(u"Пропускаем ручки: %s" % u", ".join(skip_generators))

    exporter = MultiExporter(log=log, versions=versions, skip_generators=skip_generators)
    exporter.export()
