# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import warnings

from common.models.geo import Station


warnings.filterwarnings('ignore', 'Unknown table.*')


class StationMixin:
    u"""Микс-ин для экспортов, работающих с остановками"""

    technical_stations_ids = None

    def is_technical_station(self, station, check_threads=True):
        # subway stations always shown
        if station.is_subway:
            return False

        if u'(к/ст)' in station.title:
            return True

        if not station.longitude:
            return True

        if check_threads:
            return station.rtstation_set.filter(is_technical_stop=False).count() == 0
        else:
            return False

    def station_matches_white_list_item(self, station, white_list_item):
        return (
            station.settlement_id == white_list_item.settlement_id and
            (white_list_item.t_subtype_id is None or white_list_item.t_subtype_id == station.t_subtype_id)
        )

    def all_technical_stations(self):
        if self.technical_stations_ids is None:
            self.technical_stations_ids = [
                s.id for s in Station.objects.filter(rtstation__is_technical_stop=True).distinct()
            ]

        return self.technical_stations_ids


class DataGenerator:
    """Абстрактный класс для генераторов данных"""

    @classmethod
    def insert_into_validation_table(cls, obj, conn):
        raise NotImplementedError

    def __init__(self, only_urban=False):
        self.only_urban = only_urban

    def generate(self, log):
        raise NotImplementedError


class DataFormatter:
    """Абстрактный класс для вывода строк в различных форматах"""

    NEWLINE = "\n"

    def write(self, row, file):
        file.write(self.format_string(*self.format(row)).encode('utf-8'))
        file.write(self.NEWLINE)

    def format_string(self, *args):
        u"""Отформатировать строчку для вывода по правилам"""
        return "\t".join([unicode(s or '') for s in args])

    def format(self, row):
        return row
