# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import re
import pytz
from datetime import timedelta, datetime

from common.models_abstract.schedule import ExpressType
from travel.rasp.library.python.common23.date import environment
from common.utils.date import RunMask

from travel.rasp.rasp_scripts.scripts.long_haul.export import base


td = timedelta(days=1)


def add_prefix(str, prefix='lh_', sep='__'):
    s_type, id = str.split(sep)
    return s_type + sep + prefix + id


def get_station_title(station):
    title = station.L_title(lang='ru')
    return u'Аэропорт {}'.format(title) if station.t_type.code == 'plane' else title


class RoutesFormatter(base.DataFormatter):
    u"""Экспорт маршрутов"""

    def format(self, route):
        return (
            u'lh_' + route.route_uid,
            self.export_t_type(route),
            route.accepted_thread.number,
            route.style or ''
        )

    def export_t_type(self, route):
        return route.accepted_thread.t_type.code


class RoutesFormatterAeroex(RoutesFormatter):
    u"""Экспорт маршрутов с пометкой аэроэкспрессов"""

    def export_t_type(self, route):
        is_aeroex = route.accepted_thread.express_type == ExpressType.AEROEXPRESS
        return ExpressType.AEROEXPRESS if is_aeroex else route.accepted_thread.t_type.code


class StopsFormatter(base.DataFormatter):
    def format(self, row):
        return self.export_station_or_group(row)

    def export_station_or_group(self, station):
        return (
            add_prefix(station.export_uid),
            None,
            get_station_title(station),
            'stop',
            station.latitude,
            station.longitude
        )

    def cut_brackets(self, string):
        u"""Форматирование названия станции"""
        return re.sub(r"\s+\([^)]+\)", "", string)


class ThreadsFormatter(base.DataFormatter):
    u"""Экспортер ниток. В этом классе только вспомогательный метод"""

    def format(self, thread):
        return (
            u'lh_' + thread.uid,
            u'lh_' + thread.route.route_uid,
            'circular' if thread.is_circular else 'linear',
            self.reversed_thread_id(thread),
            ''
        )

    def reversed_thread_id(self, thread):
        u"""Найти нитку, следующую в обратном направлении"""

        # не для гортранса возвращаем пустые
        if thread.route.t_type_id != 8:
            return ''

        end_points = list(thread.get_extreme_stations())
        end_points.reverse()

        for t in thread.route.rthread_set.all():
            if t == thread:
                continue

            if list(t.get_extreme_stations()) == end_points:
                return 'lh_' + t.uid

        return ''


class GeometryFormatter(base.DataFormatter):
    def format(self, pt):
        return (
            u'lh_' + pt[0],
            str(pt[1]),
            str(pt[2]),
            str(pt[3])
        )


class ThreadStopsFormatter(base.DataFormatter):
    def format(self, rtstation):
        return (
            u'lh_' + rtstation.thread.uid,
            str(rtstation.index),
            add_prefix(rtstation.station.export_uid),
            str(rtstation.arrival or 0),
            str(rtstation.departure or 0),
            str(rtstation.flags)
        )


class TimetableFormatter(base.DataFormatter):
    def format(self, row):
        return u'lh_' + row[0], row[1]


class CalendarFormatter(base.DataFormatter):
    def format(self, thread):
        now_aware = environment.now_aware()

        thread_tz_today = now_aware.astimezone(thread.pytz).date()

        # переводим календарь в UTC
        utc_datetimes = []

        for date in RunMask(thread.year_days, today=thread_tz_today).dates():
            dt = thread.pytz.localize(datetime.combine(date, thread.tz_start_time))

            utc_datetimes.append(dt.astimezone(pytz.utc))

        return (
            u'lh_' + thread.uid,
            u'lh_' + thread.uid,
            "*",
            ",".join(dt.strftime('%d.%m') for dt in utc_datetimes),
            "*",
            "*"
        )


class FrequencyFormatter:
    def format(self, row):
        pass


class TransitionsFormatter:
    def format(self, row):
        pass


class L10Formatter(base.DataFormatter):
    def format(self, row):
        return row
