# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import re
import os

from django.conf import settings
from django.db import connection as django_connection

from common.models.geo import Station, StationTerminal
from travel.rasp.rasp_scripts.scripts.long_haul.export.formatters import RoutesFormatter


DB_FILE = os.path.join(settings.EXPORT_URBAN_PUBLIC_PATH, 'validation.db')


def init_db(connection):
    cursor = connection.cursor()

    cursor.execute('''PRAGMA synchronous = 0''')
    cursor.execute('''PRAGMA journal_mode = "OFF"''')
    cursor.execute('''PRAGMA encoding = "UTF-8"''')
    cursor.execute('''PRAGMA temp_store = 2''')
    cursor.execute('''PRAGMA locking_mode = "EXCLUSIVE"''')

    cursor.execute('''
    CREATE TABLE IF NOT EXISTS validation_route (
        id integer,
        t_type_id integer,
        number  text,
        route_uid text,
        t_type text,
        style longtext
    )
    ''')
    cursor.execute('''
    CREATE TABLE IF NOT EXISTS validation_rthread (
        id integer,
        route_id integer,
        number text,
        is_circular integer,
        uid text,
        is_train integer
    )
    ''')
    cursor.execute('''
    CREATE TABLE IF NOT EXISTS validation_rtstation (
        id integer,
        thread_id integer,
        station_id integer,
        thread_uid text
    )
    ''')
    cursor.execute('''
        CREATE TABLE IF NOT EXISTS validation_station (
            id integer PRIMARY KEY AUTOINCREMENT,
            origin_id integer,
            type text,
            name text,
            export_uid text,
            latitude real,
            longitude real
        )
    ''')
    cursor.execute('''
        CREATE TABLE IF NOT EXISTS validation_timetable (
            id integer PRIMARY KEY AUTOINCREMENT,
            thread_uid text,
            time text
        )
    ''')
    cursor.execute('''
    CREATE TABLE IF NOT EXISTS validation_calendar (
        id integer PRIMARY KEY AUTOINCREMENT,
        year_days text,
        uid text,
        thread_id integer
    )
    ''')
    cursor.execute('''
    CREATE TABLE IF NOT EXISTS validation_geometry (
        id integer PRIMARY KEY AUTOINCREMENT,
        thread_id integer,
        latitude real,
        longitude real,
        thread_uid text
    )
    ''')
    cursor.execute('''
    CREATE TABLE IF NOT EXISTS rtstation_cnt (
        thread_id integer,
        cnt integer
    )
    ''')
    cursor.execute('''
    CREATE TABLE IF NOT EXISTS calendar_cnt (
        thread_id integer,
        cnt integer
    )
    ''')

    connection.commit()
    cursor.close()


def _load_geoadmin_data(conn_target, conn_source=django_connection):
    u"""Грузим данные из геоадмина для проверки"""
    cursor_source = conn_source.cursor()
    cursor_target = conn_target.cursor()

    cursor_target.execute('''SELECT DISTINCT id FROM validation_rthread''')
    threads_id = tuple([r[0] for r in cursor_target.fetchall()])
    q = '''
    SELECT thread_id, COUNT(id) as cnt
    FROM www_rtstation ts
    WHERE thread_id IN %s
    GROUP BY thread_id
    '''
    q_insert = '''INSERT INTO rtstation_cnt VALUES (?, ?)'''

    if threads_id:
        cursor_source.execute(q % str(threads_id))
        rows = cursor_source.fetchall()
    else:
        rows = []

    for row in rows:
        cursor_target.execute(q_insert, row)

    q = '''
    SELECT wt.id, COUNT(s.ts_id) as cnt
    FROM www_rthread wt JOIN (
        SELECT t.id as t_id, ts.id as ts_id
        FROM
            www_threadschedule ts INNER JOIN
            www_rthread t ON ts.thread_id = t.id INNER JOIN
            www_route r ON t.route_id = r.id LEFT JOIN
            urban_package_routes ur ON t.route_id = ur.route_id LEFT JOIN
            urban_package p ON ur.package_id = p.id
        WHERE
            p.is_exported = 1 AND t.hidden = 0
        GROUP BY ts.thread_id, ts.schedule_oid, ts.is_cancel
    ) as s ON wt.id = s.t_id
    WHERE wt.id IN %s
    GROUP BY wt.id
    '''
    q_insert = '''INSERT INTO calendar_cnt VALUES (?, ?)'''

    if threads_id:
        cursor_source.execute(q % str(threads_id))
        rows = cursor_source.fetchall()
    else:
        rows = []

    for row in rows:
        cursor_target.execute(q_insert, row)

    cursor_source.close()
    cursor_target.close()


def remove_db():
    os.unlink(DB_FILE)


def insert_route(route, connection):
    formatter = RoutesFormatter()
    sql = '''
        INSERT INTO validation_route (id, t_type_id, number, route_uid, t_type, style)
        VALUES
        (?, ?, ?, ?, ?, ?)'''
    cursor = connection.cursor()
    cursor.execute(sql, (
        route.id,
        route.accepted_thread.t_type.id,
        route.accepted_thread.number,
        route.route_uid,
        formatter.export_t_type(route),
        route.style
    ))
    connection.commit()
    cursor.close()


def insert_geometry(geometry, connection):
    sql = '''
    INSERT INTO validation_geometry (thread_id, latitude, longitude, thread_uid) VALUES (?, ?, ?, ?)
    '''
    cursor = connection.cursor()
    cursor.execute(sql, (geometry[4], geometry[2], geometry[3], geometry[0]))
    connection.commit()
    cursor.close()


def insert_thread(thread, connection):
    sql = '''INSERT INTO validation_rthread VALUES (?, ?, ?, ?, ?, ?)'''
    cursor = connection.cursor()

    if thread.route.t_type.code in ('train', 'suburban'):
        is_train = 1
    else:
        is_train = 0

    cursor.execute(sql, (thread.id, thread.route.id, thread.number, thread.is_circular, thread.uid, is_train))
    connection.commit()
    cursor.close()


def insert_thread_stop(thread_stop, connection):
    sql = '''INSERT INTO validation_rtstation VALUES (?, ?, ?, ?)'''
    cursor = connection.cursor()
    cursor.execute(sql, (thread_stop.id, thread_stop.thread.id, thread_stop.station.id, thread_stop.thread.uid))
    connection.commit()
    cursor.close()


def insert_stop(stop, connection):
    type, name, export_uid = '', '', ''

    if isinstance(stop, Station):
        type = 'stop'
        name = stop.title
        export_uid = stop.export_uid
    elif isinstance(stop, StationTerminal):
        type = 'exit'
        name = stop.name
        export_uid = stop.station.export_uid

    sql = '''
    INSERT INTO validation_station (origin_id, type, name, export_uid, latitude, longitude) VALUES
        (?, ?, ?, ?, ?, ?)
    '''
    cursor = connection.cursor()
    cursor.execute(sql, (stop.id, type, name, export_uid, stop.latitude, stop.longitude))
    connection.commit()
    cursor.close()


def insert_timetable(timetable, connection):
    cursor = connection.cursor()
    cursor.execute('''
        INSERT INTO validation_timetable (thread_uid, time) VALUES (?, ?)
    ''', (str(timetable[0]), str(timetable[1])))
    connection.commit()
    cursor.close()


def insert_calendar(calendar, connection):
    sql = '''
    INSERT INTO validation_calendar
        (year_days, uid, thread_id) VALUES
        ( ?, ?, ?)
    '''
    cursor = connection.cursor()
    cursor.execute(sql, (
        calendar.year_days,
        calendar.uid,
        calendar.id,
    ))
    connection.commit()
    cursor.close()


def process(log, conn):
    log.info(u'-- Начинаем проверку данных')

    _load_geoadmin_data(conn)
    index_tables(conn)
    validate_referential_integrity(log, conn)
    validate_overall_integrity(log, conn)
    validate_format(log, conn)
    validate_redundancy(log, conn)

    log.info(u'-- Проверка данных завершена')


def validate_referential_integrity(log, connection):
    log.info(u'-- Проверяем ссылочную целостность')
    cursor = connection.cursor()

    log.info(u'- Проверяем, что нитки ссылаются на существующие маршруты')
    query = '''
    SELECT t.id, t.route_id, t.number
    FROM
        validation_rthread t LEFT OUTER JOIN validation_route r ON t.route_id = r.id
    WHERE r.id IS NULL
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (id, route_id, number) - начало:')
    for row in rows:
        log.info(u"%d, %d, %s" % (row[0], row[1], row[2]))
    log.info(u'Проверку не прошли - конец')

    log.info(u'- Проверяем, что станции ниток ссылаются на существующие станции')
    query = '''
    SELECT ts.thread_id, ts.station_id, ts.thread_uid
    FROM
        validation_rtstation ts LEFT OUTER JOIN validation_station s ON ts.station_id = s.origin_id
    WHERE
        s.type IN ('stop', 'station') AND s.origin_id IS NULL
    '''
    cursor.execute(query)
    rows = cursor.fetchall()
    log.info(u'Проверку не прошли (thread_uid, station_id) - начало:')
    for row in rows:
        log.info(u"%d, %d" % (row[2], row[1]))
    log.info(u'Проверку не прошли - конец')

    log.info(u'- Проверяем, что станции ниток ссылаются на существующие нитки')
    query = '''
    SELECT ts.thread_id, ts.station_id, ts.thread_uid
    FROM
        validation_rtstation ts LEFT OUTER JOIN validation_rthread t ON ts.thread_id = t.id
    WHERE t.id IS NULL
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (thread_uid, station_id) - начало:')
    for row in rows:
        log.info(u"%s, %d" % (row[2], row[1]))
    log.info(u'Проверку не прошли - конец')

    log.info(u'- Проверяем, что геометрия ссылается на существующие нитки')
    query = '''
    SELECT g.thread_uid
    FROM
        validation_geometry g LEFT OUTER JOIN validation_rthread t ON g.thread_id = t.id
    WHERE t.id IS NULL
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (thread_uid) - начало:')
    for row in rows:
        log.info(u"%s" % row[0])
    log.info(u'Проверку не прошли - конец')

    log.info(u'- Проверяем, что календари ссылаются на существующие нитки')
    query = '''
    SELECT
        c.thread_id, c.uid
    FROM
        validation_calendar c LEFT OUTER JOIN validation_rthread t ON c.thread_id = t.id
    WHERE t.id IS NULL
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (uid) - начало:')
    for row in rows:
        log.info(u"%s" % (row[1]))
    log.info(u'Проверку не прошли - конец')

    log.info(u'-- Проверка ссылочной целостности завершена')


def validate_overall_integrity(log, connection):
    log.info(u'-- Проверяем общую целостность')
    cursor = connection.cursor()

    log.info(u'- Проверяем, что у всех ниток есть более одной нитки станции')
    query = '''
    SELECT t.id, t.number, t.uid
    FROM
        validation_rthread t LEFT JOIN validation_rtstation ts ON t.id = ts.thread_id
    GROUP BY t.id
    HAVING COUNT(ts.station_id) < 2
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (thread_uid, thread_number) - начало')
    for row in rows:
        log.info(u"%s, %s" % (row[2], row[1]))
    log.info(u'Проверку не прошли - конец')

    log.info(u'- Проверяем, что число станций ниток для всех ниток совпадает с их числом в геоадмине')
    sql = '''
    CREATE TABLE vrtstation_cnt AS
        SELECT vt.id as id, COUNT(vts.id) as cnt, vt.uid as uid
        FROM
            validation_rthread vt JOIN validation_rtstation vts ON vt.id = vts.thread_id
        GROUP BY vt.id
    '''
    cursor.execute(sql)
    cursor.execute('''CREATE INDEX vrt_id_idx ON vrtstation_cnt (id)''')
    cursor.execute('''CREATE INDEX vrt_cnt_idx ON vrtstation_cnt (cnt)''')

    query = '''
    SELECT s1.uid, s1.cnt
    FROM vrtstation_cnt s1 LEFT JOIN rtstation_cnt s2 ON (s1.id = s2.thread_id AND s1.cnt = s2.cnt)
    WHERE s1.cnt IS NULL
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (thread_uid, station_count) - начало:')
    for row in rows:
        log.info(u"%s, %d" % (row[0], row[1]))
    log.info(u'Проверку не прошли - конец')

    log.info(u'- Проверяем, что у всех ниток более одного узла геометрии')
    query = '''
    SELECT t.id, t.number, t.uid
    FROM
        validation_rthread t LEFT JOIN validation_geometry g ON t.id = g.thread_id
    WHERE t.is_train = 1
    GROUP BY t.id
    HAVING COUNT(g.id) < 2
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (thread_uid, thread_number) - начало:')
    for row in rows:
        log.info(u"%s, %s" % (row[2], row[1]))
    log.info(u'Проверку не прошли - конец')

    log.info(u'- Проверяем, что у всех ниток есть календари')
    query = '''
    SELECT t.id, t.number, t.uid
    FROM
        validation_rthread t LEFT JOIN validation_calendar c ON t.id = c.thread_id
    WHERE c.thread_id IS NULL
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (thread_uid, thread_number) - начало:')
    for row in rows:
        log.info(u"%s, %s" % (row[2], row[1]))
    log.info(u'Проверку не прошли - конец')

    log.info(u'-- Проверка общей целостности завершена')


def validate_format(log, connection):
    log.info(u'-- Проверяем формат')
    cursor = connection.cursor()

    log.info(u'- У остановок должны быть id, название и координаты')
    query = '''
    SELECT
        origin_id, export_uid, latitude, longitude, name
    FROM
        validation_station
    WHERE
        type IN ('stop', 'station')
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (uid, id) - начало:')
    reg = re.compile(ur'^[0-9]{2}\.[0-9]+$')
    for row in rows:
        if not row[1]:
            log.info(u'%s, %d - отсутствует ID' % (row[1], row[0]))
        elif not row[2] or not row[3]:
            log.info(u'%s, %d - отсутствуют координаты' % (row[1], row[0]))
        elif not row[4]:
            log.info(u'%s, %d - отсутствует название' % (row[1], row[0]))
        elif not reg.match(str(row[2])) or not reg.match(str(row[3])):
            log.info(u'%s, %d - неверный формат координат' % (row[1], row[0]))
    log.info(u'Проверку не прошли - конец')

    log.info(u'- У маршрутов должны быть тип и номер')
    query = '''
    SELECT id, number, t_type_id, route_uid
    FROM validation_route
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (id, uid) - начало:')
    for route_id, route_uid in [(row[0], row[3]) for row in rows if not row[1] or not row[2]]:
        log.info(u"%d, %s" % (route_id, route_uid))
    log.info(u'Проверку не прошли - конец')

    log.info(u'- У ниток должен быть тип')
    query = '''
    SELECT id, is_circular
    FROM validation_rthread
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (id) - начало:')
    for thread_id in [row[0] for row in rows if row[1] is None]:
        log.info(u"%d" % thread_id)
    log.info(u'Проверку не прошли - конец')

    log.info(u'- У календарей не должно быть пустых полей и поля дожны соответствовать формату')
    query = '''
    SELECT
        thread_id,
        year_days,
        uid
    FROM validation_calendar
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (dates) - начало:')
    for row in rows:
        if not row[0]:
            log.info(u'%s - отсутствует нитка' % (row[2]))
        if not row[1]:
            log.info(u'%s - отсутствуют дни недели' % (row[2]))

    log.info(u'Проверку не прошли - конец')

    log.info(u'- У отправлений проверяем формат времени (д.б. чч:мм:сс) ')
    query = '''
    SELECT time FROM validation_timetable
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (time) - начало:')
    row_seq = [row for row in rows if not _validate_time_format(row[0])]
    for row in row_seq:
        log.info(u"%s" % row[0])
    log.info(u'Проверку не прошли - конец')

    log.info(u'- У геометрии проверяем формат координат (д.б. xx.xxxxxx...)')
    query = '''
    SELECT thread_uid, latitude, longitude FROM validation_geometry
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (thread_uid, latitde, longitude) - начало:')
    reg = re.compile(ur'^[0-9]{2,}\.[0-9]+$')

    row_seq = [row for row in rows if not reg.match(str(row[1])) or not reg.match(str(row[2]))]
    for row in row_seq:
        log.info(u"%s, %s, %s" % (row[0], row[1], row[2]))
    log.info(u'Проверку не прошли - конец')

    log.info(u'-- Проверка формата завершена')


def validate_redundancy(log, connection):
    log.info(u'-- Проверяем избыточность')
    cursor = connection.cursor()

    log.info(u'- Проверяем, что все станции типа stop упоминаются в станциях ниток')
    query = '''
    SELECT s.origin_id
    FROM
        validation_station s LEFT OUTER JOIN validation_rtstation ts ON s.origin_id = ts.station_id
    WHERE s.type = 'stop' AND ts.station_id IS NULL
    '''
    cursor.execute(query)
    rows = cursor.fetchall()

    log.info(u'Проверку не прошли (station_id) - начало:')
    for row in rows:
        log.info(u"%d" % row[0])
    log.info(u'Проверку не прошли - конец')

    log.info(u'-- Проверка избыточности завершена')


def _validate_time_format(s, sep=':'):
    """Проверяет формат времени"""
    t = s.split(sep)
    if len(t) < 3:
        return False

    try:
        hour = int(t[0])
        minutes = int(t[1])
        seconds = int(t[2])
        range60 = range(60)

        if (
            hour not in range(24) or
            minutes not in range60 or
            seconds not in range60
        ):
            return False
    except ValueError:
        return False

    return True


def _validate_calendar_format(date):
    value = ur'([0-9]{2}(\.[0-9]{2}(\.[0-9]{4})?)?)'
    freq = '^%(value)s-%(value)s$' % {'value': value}
    seq = '^%(value)s(,%(value)s)+$' % {'value': value}
    slash_reg = '^[0-9]{1,2}/[0-9]{1,2}$'
    reg = '^%s,%s$' % (freq, slash_reg)
    any_date = ur'^\*$'

    regs = (
        re.compile('^%s$' % value),
        re.compile(freq),
        re.compile(seq),
        re.compile(slash_reg),
        re.compile(reg),
        re.compile(any_date)
    )

    if not any(bool(regex.match(date)) for regex in regs):
        return False

    return True


def index_tables(connection):
    cursor = connection.cursor()

    cursor.execute('''CREATE INDEX number_index ON validation_route (number)''')
    cursor.execute('''CREATE INDEX route_uid_index ON validation_route (route_uid)''')
    cursor.execute('''CREATE INDEX t_type_index ON validation_route (t_type)''')
    cursor.execute('''CREATE INDEX t_type_id_index ON validation_route (t_type_id)''')
    cursor.execute('''CREATE INDEX r_id_index ON validation_route (id)''')

    cursor.execute('''CREATE INDEX type_index ON validation_station (type)''')
    cursor.execute('''CREATE INDEX origin_id_index ON validation_station (origin_id)''')

    cursor.execute('''CREATE INDEX thread_number_index ON validation_rthread (number)''')
    cursor.execute('''CREATE INDEX uid_index ON validation_rthread (uid)''')
    cursor.execute('''CREATE INDEX route_id_index ON validation_rthread (route_id)''')
    cursor.execute('''CREATE INDEX t_id_index ON validation_rthread (id)''')

    cursor.execute('''CREATE INDEX time_index ON validation_timetable (time)''')
    cursor.execute('''CREATE INDEX timetable_schedule_oid_index ON validation_timetable (thread_uid)''')

    cursor.execute('''CREATE INDEX week_days_index ON validation_calendar (year_days)''')
    cursor.execute('''CREATE INDEX calendar_uid_index ON validation_calendar (uid)''')
    cursor.execute('''CREATE INDEX calendar_thread_id_index ON validation_calendar (thread_id)''')

    cursor.execute('''CREATE INDEX rtstation_station_id_index ON validation_rtstation (station_id)''')
    cursor.execute('''CREATE INDEX rtstation_thread_id_index ON validation_rtstation (thread_id)''')

    cursor.execute('''CREATE INDEX thread_cnt_index ON rtstation_cnt (thread_id)''')
    cursor.execute('''CREATE INDEX cnt_index ON rtstation_cnt (cnt)''')

    cursor.execute('''CREATE INDEX c_thread_index ON calendar_cnt (thread_id)''')
    cursor.execute('''CREATE INDEX c_cnt_index ON calendar_cnt (cnt)''')

    cursor.execute('''CREATE INDEX g_thread_id_idx ON validation_geometry (thread_id)''')
    cursor.execute('''CREATE INDEX g_thread_uid_idx ON validation_geometry (thread_uid)''')

    connection.commit()
    cursor.close()
