# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import pytz
from datetime import datetime, timedelta
from dateutil import parser
import codecs

from common.models.schedule import RThreadType
from common.models.transport import TransportType
from common.models.geo import Station
from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ, timedelta2minutes
from common.utils.tz_mask_split import (
    MaskSplitter, StationForMaskSplit, ThreadForMaskSplit, thread_mask_split, calculate_mask_for_tz_and_station
)
from common.data_api.baris.all_flights import all_flights_iterate
from common.data_api.baris.helpers import (
    make_baris_masks_from_protobuf, BarisMasksProcessor, make_pseudo_uid_for_baris_flight
)

from travel.rasp.rasp_scripts.scripts.pathfinder.helpers import get_to_pathfinder_year_days_converter


log = logging.getLogger(__name__)


def _schedule_is_valid(schedule, station_time_zones):
    for point in schedule.Route:
        if point.AirportID not in station_time_zones:
            return False
    return True


class ThegraphStation(StationForMaskSplit):
    def __init__(
        self, station_id, station_pytz, arrival_time, arrival_day_shift, departure_time, departure_day_shift
    ):
        super(ThegraphStation, self).__init__(
            station_pytz, arrival_time, arrival_day_shift, departure_time, departure_day_shift
        )
        self.station_id = station_id


def _make_baris_thegraph_thread(schedule, stations_pytz_dict):
    """
    Подготовка объекта ThreadForMaskSplit для разбеиения маски и для формирования записи в thegraph
    """
    start_tz = stations_pytz_dict[schedule.Route[0].AirportID]
    baris_masks = make_baris_masks_from_protobuf(schedule)
    mask_processor = BarisMasksProcessor(baris_masks, start_tz)

    run_mask = mask_processor.run_mask
    stations = [
        ThegraphStation(
            station_id=point.AirportID,
            station_pytz=stations_pytz_dict[point.AirportID],
            arrival_time=parser.parse(point.ArrivalTime).time() if point.ArrivalTime else None,
            arrival_day_shift=point.ArrivalDayShift,
            departure_time=parser.parse(point.DepartureTime).time() if point.DepartureTime else None,
            departure_day_shift=point.DepartureDayShift,
        )
        for point in schedule.Route
    ]

    return ThreadForMaskSplit(run_mask, stations)


def _get_mask_last_run_day(run_mask):
    last_day = None
    for day in run_mask.iter_dates(past=True):
        last_day = day

    return last_day


def _get_moscow_dt(nearest_date, init_pytz, time, days_shift):
    if time is None:
        return None

    naive_dt = datetime.combine(nearest_date, time) + timedelta(days=days_shift)
    return init_pytz.localize(naive_dt).astimezone(MSK_TZ)


def _make_thread_schedule_props(flight_number, airline_id, run_mask):
    last_date = _get_mask_last_run_day(run_mask)
    if not last_date:
        return

    flight_number_decoded = flight_number.decode('utf-8')
    # В БАРиС нет uid-ов для ниток, поэтому генерируем псевдо-uid
    pseudo_uid = make_pseudo_uid_for_baris_flight(flight_number_decoded, airline_id, last_date)

    return pseudo_uid, last_date


def _iterate_thread_thegraph_rows(thegraph_thread, pseudo_uid, flight_number, last_date, run_mask, year_days_func):
    """
    Формируем список записей thegraph для одной маски дней вылета одного рейса
    :param thegraph_thread: нитка, объект типа ThreadForMaskSplit со станциями типа ThegraphStation
    :param pseudo_uid: псевдо-uid авиарейса
    :param flight_number: номер рейса
    :param last_date: последний день вылета
    :param run_mask: маска дней вылета
    :param year_days_func: функция формирования списка дней вылета по маске
    """

    flight_number_decoded = flight_number.decode('utf-8')

    result_rows = []
    is_correct_thread = True

    for from_index in range(0, len(thegraph_thread.stations) - 1):
        from_station = thegraph_thread.stations[from_index]
        to_station = thegraph_thread.stations[from_index + 1]
        from_id = from_station.station_id
        to_id = to_station.station_id

        from_pytz = from_station.pytz
        from_arrival_moscow_dt = _get_moscow_dt(
            last_date, from_pytz, from_station.arrival_time, from_station.arrival_day_shift
        )
        from_departure_moscow_dt = _get_moscow_dt(
            last_date, from_pytz, from_station.departure_time, from_station.departure_day_shift
        )

        to_pytz = to_station.pytz
        to_arrival_moscow_dt = _get_moscow_dt(
            last_date, to_pytz, to_station.arrival_time, to_station.arrival_day_shift
        )

        from_stay_duration = (
            int(timedelta2minutes(from_departure_moscow_dt - from_arrival_moscow_dt))
            if from_arrival_moscow_dt else 0
        )
        trip_duration = int(timedelta2minutes(to_arrival_moscow_dt - from_departure_moscow_dt))

        if from_stay_duration < 0 or trip_duration <= 0:
            is_correct_thread = False

        from_departure_mask = calculate_mask_for_tz_and_station(
            run_mask, from_station.departure_time, from_station.departure_day_shift, from_pytz, MSK_TZ
        )

        result_rows.append((
            from_id,  # id аэропорта отправления
            to_id,  # id аэропорта прибытия
            0,  # fuzzy, для самолетов всегда = 0
            pseudo_uid,  # Замена uid для ниток из БАРиС
            from_departure_moscow_dt.time(),  # Время отправления в московской таймзоне
            from_stay_duration,  # Время стоянки в аэропорту отправления
            trip_duration,  # Время полета до конечной станции
            flight_number_decoded,  # Номер рейса
            TransportType.PLANE_ID,  # Тип транспорта
            RThreadType.BASIC_ID,  # Тип нитки
            year_days_func(from_departure_mask)  # Маска дней вылета в московской таймзоне
        ))

    if not is_correct_thread:
        log.error('Negative trip or stay duration: {}'.format(pseudo_uid))
    else:
        for row in result_rows:
            yield map(unicode, row)


def _iterate_on_baris_all_flights():
    station_list = Station.objects.filter(t_type_id=TransportType.PLANE_ID).values_list('id', 'time_zone')
    stations_pytz_dict = {station_id: pytz.timezone(time_zone) for station_id, time_zone in station_list}

    year_days_func = get_to_pathfinder_year_days_converter(environment.today())
    mask_splitter = MaskSplitter()

    flight_numbers = set()
    flight_uids = set()
    for flight in all_flights_iterate():
        if flight.Title in flight_numbers:
            log.error('Duplicated flight number: {}'.format(flight.Title))
            continue

        flight_numbers.add(flight.Title)

        for schedule in flight.Schedules:
            if not _schedule_is_valid(schedule, stations_pytz_dict):
                continue

            try:
                thegraph_thread = _make_baris_thegraph_thread(schedule, stations_pytz_dict)
            except Exception as ex:
                log.exception(ex.message)
                continue

            moscow_invariant_masks = thread_mask_split(thegraph_thread, mask_splitter, MSK_TZ)

            for run_mask in moscow_invariant_masks:
                last_date = _get_mask_last_run_day(run_mask)
                if not last_date:
                    continue

                flight_number_decoded = flight.Title.decode('utf-8')
                # В БАРиС нет uid-ов для ниток, поэтому генерируем псевдо-uid
                pseudo_uid = make_pseudo_uid_for_baris_flight(flight_number_decoded, flight.AirlineID, last_date)

                if pseudo_uid in flight_uids:
                    # Рейсы с повторяющимся uid не добавляем, иначе пересадочнику плохо становится
                    log.error('Duplicated flight pseudo-uid: {}'.format(pseudo_uid))
                else:
                    flight_uids.add(pseudo_uid)
                    for row in _iterate_thread_thegraph_rows(
                        thegraph_thread, pseudo_uid, flight.Title, last_date, run_mask, year_days_func
                    ):
                        yield row


def gen_thegraph_from_baris(thegraph_path):
    """
    Генерация файла thegraph из БАРиС
    """
    log.info('Generation of thegraph from BARiS')
    try:
        with codecs.open(thegraph_path, 'a', encoding='utf-8') as result_file:
            count = 0
            for row in _iterate_on_baris_all_flights():
                result_file.write('\t'.join(row))
                result_file.write('\n')

                count += 1
                if count % 5000 == 0:
                    log.info('{} segments added'.format(count))

    except Exception as ex:
        log.exception(u'Error during the generation from BARiS. {}'.format(ex.message))
        raise

    log.info('Generation of thegraph from BARiS completed. Totally {} segments added'.format(count))
