# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
import codecs
from datetime import datetime, timedelta

import pytz
from django.conf import settings

from common.db.switcher import get_connection_by_role
from common.models.currency import Currency
from common.models.geo import Station
from common.models.tariffs import ThreadTariff
from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ, daterange, timedelta2minutes
from common.models_admin.schedule import StationTeleportMap

from travel.rasp.rasp_scripts.scripts.pathfinder.helpers import get_to_pathfinder_year_days_converter
from travel.rasp.rasp_scripts.scripts.pathfinder.originalbox import OriginalBox


log = logging.getLogger(__name__)


class Utilities:
    @classmethod
    def raw_sql_to_file(cls, sql, data_path, filename):
        cursor = get_connection_by_role().cursor()
        cursor.execute(sql)

        with codecs.open(os.path.join(data_path, filename), 'w', 'utf-8') as f:
            for row in cursor.fetchall():
                f.write('\t'.join(map(unicode, row)) + '\n')


def _gen_tz_deltas(tz, start_date, days_in_year):
    jan1 = datetime(start_date.year, 1, 1, 12, 0, 0)
    day = datetime(start_date.year, start_date.month, start_date.day, 12, 0, 0)

    res = [0] * days_in_year
    for i in range(0, days_in_year):
        day += timedelta(1)
        delta = tz.utcoffset(day) - MSK_TZ.utcoffset(day)
        res[(day - jan1).days % days_in_year] = int(timedelta2minutes(delta))

    return res


def _fill_time_zone_map(today, file_path):
    log.info(u'Generation of the file time_zone.map.')

    start_date = today - timedelta(days=settings.DAYS_TO_PAST)
    days_in_year = 366 if any(mask_date.month == 2 and mask_date.day == 29
                              for mask_date in daterange(start_date, start_date + timedelta(days=365))) else 365

    with codecs.open(file_path, 'w', 'utf-8') as time_zone_file:
        for time_zone in Station.objects.values_list('time_zone', flat=True).distinct():
            if not time_zone:
                continue
            print(time_zone, *_gen_tz_deltas(pytz.timezone(time_zone), start_date, days_in_year), file=time_zone_file)


def _teleport_to_row(teleport):
    return (teleport.station_from_id or teleport.settlement_from_id,
            teleport.station_to_id or teleport.settlement_to_id,
            teleport.teleport_time)


def _fill_station_teleport_map(file_path):
    log.info(u'Generation of the file station_teleport.map.')

    with codecs.open(file_path, 'w', 'utf-8') as station_teleport_file:
        for teleport in StationTeleportMap.objects.exclude(station_from__isnull=True, settlement_from__isnull=True) \
                                                  .exclude(station_to__isnull=True, settlement_to__isnull=True):
            print(*_teleport_to_row(teleport), file=station_teleport_file)


ThreadTariffBox = OriginalBox('thread_uid', 'station_from_id', 'station_to_id', 'tariff', 'year_days', 'currency')


def _tariff_to_file_row(tariff, to_pathfinder_year_days):
    return (tariff.thread_uid,
            tariff.station_from_id,
            tariff.station_to_id,
            tariff.tariff,
            to_pathfinder_year_days(tariff.year_days),
            tariff.currency)


def _gen_threadtariff_file(today, file_path):
    log.info(u'Generation of the file threadtariff.map')

    to_pathfinder_year_days = get_to_pathfinder_year_days_converter(today)

    with codecs.open(file_path, 'w', 'utf-8') as threadtariff_file:
        for tariff in ThreadTariffBox.iter_queryset_chunked(ThreadTariff.objects.all()):
            print(*_tariff_to_file_row(tariff, to_pathfinder_year_days), sep='\t', file=threadtariff_file)


def generate_currency_map(data_path):
    currencies = Currency.objects.all().order_by()
    geoid = 213

    src, rates = Currency.fetch_rates(currencies, geoid)
    frates = open(os.path.join(data_path, "currency-rates.map"), 'w')
    for cur, rate in rates.iteritems():
        frates.write(cur + '\t' + unicode(rate) + '\n')
    frates.close()


def generate_map_files(data_path):
    log.info(u'Data for pathfinder generation started')

    dictionaries = (
        ('townOrig-name.map',
         "SELECT t.id, IF(t.title IS NULL OR t.title = '', \'NULL\', t.title) FROM www_settlement t ORDER BY t.id"),
        ('station-table.map',
         '''
         SELECT t.id, IF (t.title IS NULL OR t.title = '', \'NULL\', t.title), t.settlement_id, t.majority_id,
             t.country_id, t.time_zone
         FROM www_station t
         UNION SELECT t.id, t.title, s.settlement_id, t.majority_id, t.country_id, t.time_zone
         FROM www_station t, www_station2settlement s
         WHERE s.station_id = t.id'''),
        ('stationOrig-geo.map', 'select t.id, t.latitude, t.longitude from www_station t order by t.id'),
        ("stationOrig-expressCode.map",
         """
         SELECT sc.station_id, sc.code
         FROM www_stationcode sc
         JOIN www_codesystem cs ON sc.system_id = cs.id
         WHERE cs.code = 'express'"""),
        ('aeroextariff.map',
         "SELECT station_from_id, station_to_id, tariff, reverse, type_id"
         " FROM www_aeroextariff WHERE type_id=1 OR type_id=2"),
        ('interval-threads.map',
         "SELECT uid, begin_time, end_time, density, time_zone FROM www_rthread WHERE type_id=12")
    )

    today = environment.today()

    for filename, sql in dictionaries:
        log.info(u'Generation of the file %s' % filename)
        Utilities.raw_sql_to_file(sql, data_path, filename)

    _fill_time_zone_map(today, os.path.join(data_path, 'time-zone.map'))
    _fill_station_teleport_map(os.path.join(data_path, 'station_teleport.map'))
    _gen_threadtariff_file(today, os.path.join(data_path, 'threadtariff.map'))
