# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import logging
import os
from datetime import timedelta

import yt.wrapper
from django.conf import settings

from travel.rasp.library.python.common23.date import environment
from common.data_api.yt.instance import update_yt_wrapper_config


log = logging.getLogger('dump_yt_price_to_pathfinder')


def run():
    try:
        data_path = settings.PATHFINDER_DATA_PATH
        if not os.path.exists(data_path):
            os.makedirs(data_path)

        day_ago_date_str = (environment.today() - timedelta(days=1)).strftime("%Y-%m-%d")
        today_date_str = environment.today().strftime("%Y-%m-%d")
        destination_dir = os.path.join(settings.YT_ROOT_PATH, settings.ENVIRONMENT, 'rasp-min-prices-some-days')
        day_ago_dump_table = os.path.join(destination_dir, day_ago_date_str)
        today_dump_table = os.path.join(destination_dir, today_date_str)

        update_yt_wrapper_config()

        # Таблица с ценами в Yt может быть еще не готова к моменту запуска задачи
        # Таблица создается в tasks в 03:10 мск https://a.yandex-team.ru/arc/trunk/arcadia/travel/rasp/tasks/job_config.py?rev=6889375#L32
        # Если таблицы еще нет, то используем вчерашнюю
        # Если есть, то ждем конца записи в нее
        dump_table = None

        if yt.wrapper.exists(today_dump_table):
            try:
                with yt.wrapper.Transaction():
                    # exclusive лок ставится при записи новой таблицы
                    # его нельзя повторно взять под другой транзакцией
                    # пытаемся получить лок, чтобы не прочитать данные во время записи
                    yt.wrapper.lock(today_dump_table, waitable=True, wait_for=1000 * 60 * 20, mode='exclusive')
                    dump_table = today_dump_table
            except Exception:
                log.error('Wait lock exception')

        if not dump_table and yt.wrapper.exists(day_ago_dump_table):
            dump_table = day_ago_dump_table

        if not dump_table:
            raise Exception("dump_yt_price_to_pathfinder failed. tables don't exist")

        with open(os.path.join(data_path, 'dump'), 'w') as dump:
            for row in yt.wrapper.read_table(dump_table, format='dsv', raw=True):
                dump.write(row)
    except Exception:
        log.exception('dump_yt_price_to_pathfinder exception:')


def main():
    run()


if __name__ == '__main__':
    main()
