# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

"""
get_tmp_dir
get_tmp_filepath - позволяют получить времееный файл, или папку.
    Папка будет жить пока процесс не завершиться.
    При форке, новый процесс получает свою папку, которая не удалиться при завершении родителя.

    Если процесс убит с -9, то папка останется.
    Если был форк и потомок завершен с os._exit() нужно вручную очистить папку вызвав clean


clean_temp_context - исползуется с with очищает временные файлы, используемые в контексте

clean_temp - декоратор очищает временные файлы, используемые в функции
"""

import os
import os.path
import tempfile
from travel.rasp.library.python.common23.date import environment
from functools import wraps
from shutil import rmtree
from contextlib import contextmanager


from django.conf import settings
from django.utils.functional import cached_property


class TempStorage(object):
    def __init__(self):
        self.tmp_dirs = []
        self.tmp_files = []
        self.base_created = False

    @cached_property
    def base(self):
        prefix = 'rasp-{}-{}-'.format(
            settings.CURRENT_USER_NAME.split('@')[0],
            environment.now().strftime('%Y%m%d%H%M%S')
        ).replace('/', '-')

        base_dir = tempfile.mkdtemp(prefix=prefix)

        self.base_created = True

        return base_dir

    def get_tmp_dir(self, subpath=None):
        if subpath:
            subpath = os.path.abspath('/' + subpath).lstrip('/')  # удаляем точки
            parts = filter(None, os.path.split(subpath))
        else:
            parts = []

        if len(parts) > 1:
            base_tmp_dir = os.path.join(self.base, *parts[:-1])
            prefix = parts[-1] + '-'
        elif len(parts) == 1:
            base_tmp_dir = self.base
            prefix = parts[0] + '-'
        else:
            base_tmp_dir = self.base
            prefix = ''

        if not os.path.exists(base_tmp_dir):
            os.makedirs(base_tmp_dir)

        tmp_dir = tempfile.mkdtemp(prefix=prefix, dir=base_tmp_dir)

        if isinstance(tmp_dir, unicode):
            tmp_dir = tmp_dir.encode('utf8')

        self.tmp_dirs.append(tmp_dir)

        return tmp_dir

    def get_tmp_filepath(self, filename=None, subpath=None):
        if subpath:
            subpath = os.path.abspath('/' + subpath).lstrip('/')  # удаляем точки
            base_tmp_dir = os.path.join(self.base, subpath)
        else:
            base_tmp_dir = self.base

        if not os.path.exists(base_tmp_dir):
            os.makedirs(base_tmp_dir)

        if isinstance(base_tmp_dir, unicode):
            base_tmp_dir = base_tmp_dir.encode('utf8')

        if isinstance(filename, unicode):
            filename = filename.encode('utf8')

        if filename:
            (fd, filepath) = tempfile.mkstemp(prefix=filename + '-', dir=base_tmp_dir)
        else:
            (fd, filepath) = tempfile.mkstemp(dir=base_tmp_dir)

        os.close(fd)

        self.tmp_files.append(filepath)

        return filepath

    def close(self):
        if getattr(settings, 'CLEAN_TMP', True) and self.base_created:
            rmtree(self.base, ignore_errors=True)


class TempManager(object):
    def __init__(self):
        self.interprocess_tempstorages = {
            os.getpid(): [TempStorage()]
        }

    def current_process_temp_storages(self):
        return self.interprocess_tempstorages.setdefault(os.getpid(), [])

    def get_tmp_dir(self, subpath=None):
        temp_storage = self.get_current_storage()

        return temp_storage.get_tmp_dir(subpath)

    def get_tmp_filepath(self, filename=None, subpath=None):
        temp_storage = self.get_current_storage()

        return temp_storage.get_tmp_filepath(filename, subpath)

    def get_current_storage(self):
        tempstorages = self.current_process_temp_storages()
        if not tempstorages:
            raise Exception('Temp Storage was not created')

        return tempstorages[-1]

    def push(self):
        self.current_process_temp_storages().append(TempStorage())

    def pop(self):
        tempstorages = self.current_process_temp_storages()
        if tempstorages:
            temp_storage = tempstorages.pop()
            temp_storage.close()


temp_manager = TempManager()


@contextmanager
def clean_temp_context():
    temp_manager.push()
    try:
        yield temp_manager.get_current_storage()
    finally:
        temp_manager.pop()


def clean_temp_decorator(func):
    @wraps(func)
    def decorator(*args, **kwargs):
        temp_manager.push()
        try:
            return func(*args, **kwargs)
        finally:
            temp_manager.pop()

    return decorator


def clean_temp(func=None):
    if func is not None:
        return clean_temp_decorator(func)

    else:
        return clean_temp_context()


clean_tmpfiles = clean_temp


def get_tmp_dir(subpath=None):
    return temp_manager.get_tmp_dir(subpath)


def get_tmp_filepath(filename=None, subpath=None):
    return temp_manager.get_tmp_filepath(filename, subpath)


@contextmanager
def temporary_directory(subpath=None):
    with clean_temp_context():
        yield temp_manager.get_tmp_dir(subpath)
