# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import hashlib
import logging

from common.cysix.builder import ChannelBlock, GroupBlock, ThreadBlock, ScheduleBlock, StoppointBlock
from common.db.mds.clients import mds_s3_common_client
from travel.rasp.library.python.common23.date.environment import today
from rasp_vault.api import get_secret
from travel.rasp.library.python.api_clients.chelyabinsk_bus import ChelyabinskBusClient

log = logging.getLogger(__name__)


def title_hash(title):
    return hashlib.md5(title.encode()).hexdigest()


class MissedTitleError(Exception):
    pass


def build_thread_stoppoints(group_block, points, fare):
    st_from = None
    stoppoints = []

    for index in range(len(points)):
        point = points[index]

        try:
            title = point['Point']
            if not title:
                raise MissedTitleError('Missed stop point title. Stop point is skipped')

            station = group_block.add_station(title, title_hash(title), 'local')

            if index == 0:
                st_from = station

            stoppoints.append({
                'station': station,
                'arrival': point['ArrivalTime'][11:] if stoppoints else '',
                'departure': point['DepartureTime'][11:],
                'distance': point['DistanceFromTheFirstStop']
            })

            price = point.get('TicketPrice')
            if price:
                fare.add_price_block(
                    price_value=price,
                    currency='RUR',
                    stop_from=st_from,
                    stop_to=station,
                    oneway_fare=True
                )

        except MissedTitleError as ex:
            if index == 0:
                log.warning('Error in the first thread stop: {}'.format(repr(ex)))
                break
            elif index == len(points) - 1:
                log.warning('Error in the last thread stop: {}'.format(repr(ex)))
                stoppoints = []
                break
            else:
                log.warning(repr(ex))

    return stoppoints


def build_group_threads(bus_schedule, all_threads_titles, group_block):
    result = dict()

    for date, threads in bus_schedule:
        for thread in [t for t in threads if t['Flight']]:
            thread_title = thread['Flight'][10:].strip()
            if thread_title in result:
                result[thread_title]['dates'].add(date)
                continue

            if thread_title in all_threads_titles:
                continue
            else:
                all_threads_titles.add(thread_title)

            fare = group_block.add_local_fare()
            carrier = thread['AutotransportCompany']
            vehicle = thread['ModelBus']

            title = ' '.join(thread_title.split(' ')[1:]).strip()
            route_number = thread['RoutingNumber']

            stoppoins = build_thread_stoppoints(group_block, thread['TrackList'], fare)
            if not stoppoins:
                log.warning('Error in stop points for the thread {}. Thread is skipped'.format(thread_title))
                continue

            thread_data = {
                'title': title,
                'number': route_number,
                'fare': fare,
                'stoppoints': stoppoins,
                'dates': {date}
            }
            if vehicle:
                thread_data['vehicle'] = group_block.add_vehicle(vehicle, title_hash(vehicle), 'local')
            else:
                log.warning('Vehicle is missed for the thread {}'.format(title))

            if carrier:
                thread_data['carrier'] = group_block.add_carrier(carrier, title_hash(carrier), 'local')
            else:
                log.warning('Carrier is missed for the thread {}'.format(title))

            log.info('Thread {} is built'.format(title))
            result[thread_title] = thread_data

    return result.values()


def build_schedule(client, days_count):
    channel_block = ChannelBlock('bus', timezone='local')
    all_thread_titles = set()

    buses = client.get_buses()
    log.info('Get {} buses'.format(len(buses)))

    for bus in buses:
        if bus['name'] is None:
            log.warn('Unknown bus : {}'.format(str(bus)))
            continue

        group_block = GroupBlock(channel_block, title=bus['name'], code=bus['id'], t_type='bus')
        channel_block.add_group_block(group_block)

        log.info('Build schedule for {}, id={}'.format(bus['name'], bus['id']))
        bus_schedule = client.get_schedule(bus['id'], today(), days_count)
        threads = build_group_threads(bus_schedule, all_thread_titles, group_block)

        log.info('Built {} threads for {}'.format(len(threads), bus['name']))

        for thread in threads:
            thread_block = ThreadBlock(
                group_block,
                title=thread['title'],
                number=thread['number'],
                vehicle=thread['vehicle'],
                carrier=thread['carrier'],
                fare=thread['fare'],
            )

            for point in thread['stoppoints']:
                stoppoint_block = StoppointBlock(
                    thread=thread_block,
                    station=point['station'],
                    arrival_time=point['arrival'],
                    departure_time=point['departure'],
                    distance=point['distance']
                )

                thread_block.add_stoppoint_block(stoppoint_block)

            for date in thread['dates']:
                schedule_block = ScheduleBlock(thread_block, date)
                thread_block.add_schedule_block(schedule_block)

            group_block.add_thread_block(thread_block)

    return channel_block.to_unicode_xml()


def run(**kwargs):
    login = get_secret('rasp-common.RASP_CHELYABINSK_BUS_LOGIN')
    password = get_secret('rasp-common.RASP_CHELYABINSK_BUS_PASSWORD')
    days_count = kwargs.get('days_count', 2)

    client = ChelyabinskBusClient(login=login, password=password)
    schedule_xml = build_schedule(client, days_count=days_count)

    mds_path = kwargs.get('mds_path', 'buses/chelyabinsk_rasp.xml')
    mds_s3_common_client.save_data(mds_path, schedule_xml)

    log.info('Schedule uploaded to {}'.format(mds_path))


if __name__ == '__main__':
    run()
