# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from datetime import date
from typing import AnyStr, Dict, List

from lxml import etree

from travel.rasp.library.python.api_clients.krasnodar_bus import KrasnodarBusClient
from travel.rasp.rasp_scripts.scripts.schedule.krasnodar.models import Destination, Passage


log = logging.getLogger()


class KrasnodarBusParser(object):
    def __init__(self, client, availiable_avs):
        # type: (KrasnodarBusClient, Dict[AnyStr, Destination]) -> KrasnodarBusParser

        self._client = client
        self._availiable_avs = availiable_avs

    def get_destinations(self, kodp):
        # type: (Destination) -> List[Destination]

        try:
            destinations = self._client.dest(kodp.id)
        except Exception:
            return []
        try:
            return [
                Destination(id=str(destination.attrib['id']), name=destination.text.strip())
                for destination in destinations
            ]
        except KeyError:
            log.warning('Key error:\n{}'.format(etree.tostring(destinations, encoding=unicode, pretty_print=True)))
            return []

    def get_passages(self, dt, kodp, kpp):
        # type: (date, Destination, Destination) -> List[Passage]

        # API даёт доступ на запрос только от станций из списка AVAILIABLE_AVS
        if kodp.id not in self._availiable_avs:
            return []

        # Иногда API возвращает 200 без корня xml. Такие запросы необходимо "заретраить".
        # Это тяжело сделать через http-клиент, потому что на уровне http всё ОК.
        #
        # При реальных ошибках на стороне партнёра API может таймаутить или сваливаться с 5xx -
        # это обрабатывается на уровне http-клиента.
        passages = None
        for _ in range(3):
            try:
                passages = self._client.reis(dt, kodp.id, kpp.id)
                break
            except Exception:
                continue
        if passages is None:
            return []

        # Когда мы делаем запрос между станциями на дату, в которую нет маршрутов - апи отвечает 200 кодом
        # и сообщением с текстом ошибки.
        error = passages.find('error')
        if error is not None:
            log.warning('Error for reises `{}`->`{}` ({}): {}'.format(
                kodp.name,
                kpp.name,
                dt.strftime('%d:%m:%y'),
                error.attrib['code'])
            )
            return []
        try:
            return [
                Passage(
                    id=passage.attrib['id'],
                    name=passage.find('name').text.strip(),
                    statement=passage.find('statement').text,
                    atp=passage.find('atp').text.strip(),
                    busmark=passage.find('busmark').text.strip(),
                    bustype=passage.find('bustype').text.strip(),
                    totalseats=passage.find('totalseats').text,
                    freeseats=passage.find('freeseats').text,
                    departure=passage.find('departure').text,
                    arrival=passage.find('arrival').text,
                    tariff=passage.find('tariff').text,
                    preftariff=passage.find('preftariff').text,
                    bagtariff=passage.find('bagtariff').text,
                    category=passage.find('category').text,
                    tranzit=passage.find('tranzit').text,
                    saleseatlist=[int(x.text) for x in passage.find('saleseatlist')],
                    search_dest_from=kodp.id,
                    search_dest_to=kpp.id
                ) for passage in passages
            ]
        except KeyError:
            log.warning('Key error:\n{}'.format(etree.tostring(passages, encoding=unicode, pretty_print=True)))
            return []
