# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import pickle
from datetime import date
from typing import AnyStr, List, Optional

from travel.rasp.rasp_scripts.scripts.schedule.krasnodar.models import Destination, Passage


class DestinationStorage(object):
    DUMP_NAME = 'destinations'

    def __init__(self, storage_by_id=None, response_cache=None):
        self._storage_by_id = {} if storage_by_id is None else storage_by_id
        self._response_cache = {} if response_cache else response_cache

    def add(self, destination):
        # type: (Destination) -> None
        self._storage_by_id[destination.id] = destination

    def add_response(self, kodp, destinations_list):
        # type: (Destination, List[Destination]) -> None
        ids = []
        for dest in destinations_list:
            self.add(dest)
            ids.append(dest.id)
        self._response_cache[kodp.id] = ids

    def get_by_id(self, _id):
        # type: (AnyStr) -> Optional[Destination]
        return self._storage_by_id.get(_id)

    def get_response(self, kodp):
        # type: (Destination) -> Optional[List[Destination]]
        dest_ids = self._response_cache.get(kodp.id)
        if dest_ids is None:
            return None
        return [self.get_by_id(dest_id) for dest_id in dest_ids]

    def dump(self):
        with open(self.DUMP_NAME, 'wb') as f:
            f.write(pickle.dumps({
                '_storage_by_id': dict(self._storage_by_id),
                '_response_cache': dict(self._response_cache)
            }))

    def load(self):
        with open(self.DUMP_NAME, 'rb') as f:
            destinations = pickle.loads(f.read())
        self._storage_by_id = destinations['_storage_by_id']
        self._response_cache = destinations['_response_cache']


class PassageStorage(object):
    DUMP_NAME = 'passages'

    def __init__(self, response_cache=None):
        self._response_cache = {} if response_cache is None else response_cache

    def add_response(self, st_from, st_to, dt, passages):
        # type: (Destination, Destination, date, List[Passage]) -> None
        self._response_cache[(st_from.id, st_to.id, dt)] = passages

    def get_response(self, st_from, st_to, dt):
        # type: (Destination, Destination, date) -> Optional[List[Passage]]
        return self._response_cache.get((st_from.id, st_to.id, dt))

    def dump(self):
        with open(self.DUMP_NAME, 'wb') as f:
            f.write(pickle.dumps(dict(self._response_cache)))

    def load(self):
        with open(self.DUMP_NAME, 'rb') as f:
            self._response_cache = pickle.loads(f.read())
