#include "airport_panel.h"

using namespace NRasp;

TFlights TAirportPanel::FetchFlights(
    const TStationWrapper& airport,
    const TSegmentsWithRange& segmentsWithRange,
    const ui16 segmentsLimit,
    const TCommonSegmentFinder::EEventType eventType) const {
    const auto [segmentsPtr, range] = segmentsWithRange;
    const auto [beginIndex, endIndex] = range;

    if (segmentsPtr == nullptr || beginIndex == endIndex) {
        return {};
    }

    const auto indexOffset = (StatusesLimit - segmentsLimit) / 2;
    const auto& segments = *segmentsPtr;

    if (indexOffset >= 0) {
        // расширяем спискок сегментов
        const auto statusBeginIndex = std::max(0, static_cast<i32>(beginIndex - indexOffset));
        const auto statusEndIndex = std::min(segments.size(), beginIndex + segmentsLimit + indexOffset + 1);
        const auto statusesSegments =
            MakeArrayRef(segments.begin() + statusBeginIndex, segments.begin() + statusEndIndex);
        return StatusesFetcher.Fetch(airport, statusesSegments, eventType);

    } else {
        // запрашиваем статусы только по первым StatusesLimit сегментам
        const auto statusEndIndex = std::min(segments.size(), beginIndex + StatusesLimit);
        const auto statusesSegments =
            MakeArrayRef(segments.begin() + beginIndex, segments.begin() + statusEndIndex);
        const auto scheduleSegments =
            MakeArrayRef(segments.begin() + std::min(statusEndIndex + 1, endIndex - 1), segments.begin() + endIndex);

        auto flights = StatusesFetcher.Fetch(airport, statusesSegments, eventType);
        for (const auto& segment : scheduleSegments) {
            flights.emplace_back(&segment, Nothing());
        }
        return flights;
    }
}

void TAirportPanel::FilterFlights(
    TFlights& flights,
    const TFlightEventInstantGetter& getEventInstant,
    const TInstant& minEventInstant,
    const TInstant& maxEventInstant) const {
    SortBy(flights, getEventInstant);

    const auto predicate = [&getEventInstant](const TFlight& flight, const TInstant& eventInstant) {
        return getEventInstant(flight) < eventInstant;
    };
    flights.erase(flights.begin(), std::lower_bound(flights.begin(), flights.end(), minEventInstant, predicate));
    flights.erase(std::lower_bound(flights.begin(), flights.end(), maxEventInstant, predicate), flights.end());
}

TFlights TAirportPanel::MakeFlightsWithoutStatuses(const TSegmentsWithRange& segmentsWithRange) const {
    auto result = NRasp::TFlights{};

    const auto [segmentsPtr, range] = segmentsWithRange;
    const auto [beginIndex, endIndex] = range;
    const auto& segments = *segmentsPtr;

    for (size_t i = beginIndex; i < endIndex; i++) {
        result.emplace_back(&segments[i], Nothing());
    }

    return result;
}

TAirportPanelFlights TAirportPanel::FindFlights(const TStationQuery& query) const {
    const auto preparedQuery = query.ToInnerIds(SearchIndex);
    const auto& airport = SearchIndex.GetItemWithId<TStationWrapper>(preparedQuery.StationId);
    const auto [departureSegmentsWithRange, arrivalSegmentsWithRange] = StationSegmentsIndex.GetSegments(preparedQuery);

    if (AllowedStationIds.contains(query.StationId)) {
        auto departureFlights = FetchFlights(airport,
                                             departureSegmentsWithRange, query.Limit,
                                             TCommonSegmentFinder::EEventType::Departure);
        auto arrivalFlights = FetchFlights(airport,
                                           arrivalSegmentsWithRange, query.Limit,
                                           TCommonSegmentFinder::EEventType::Arrival);

        const auto minEventInstant = NDatetime::ToAbsoluteTime(query.MinEventTM, airport.Timezone());
        const auto maxEventInstant = NDatetime::ToAbsoluteTime(query.MaxEventTM, airport.Timezone());

        FilterFlights(departureFlights, TFlight::GetDepartureInstant, minEventInstant, maxEventInstant);
        FilterFlights(arrivalFlights, TFlight::GetArrivalInstant, minEventInstant, maxEventInstant);

        return {departureFlights, arrivalFlights};
    } else {
        const auto departureFlights = MakeFlightsWithoutStatuses(departureSegmentsWithRange);
        const auto arrivalFlights = MakeFlightsWithoutStatuses(arrivalSegmentsWithRange);

        return {departureFlights, arrivalFlights};
    }
}
