#pragma clang diagnostic ignored "-Wunused-parameter"

#include "config.h"

#include <util/system/info.h>

using namespace NRasp;

const THashMap<TString, TRThread::ETransportType> TConfig::Types = {
    {"bus", TRThread::BUS},
    {"train", TRThread::TRAIN},
    {"plane", TRThread::PLANE},
    {"suburban", TRThread::SUBURBAN},
    {"water", TRThread::WATER},
    {"helicopter", TRThread::HELICOPTER}};

TConfig TConfig::FromFile(const TString& path) {
    using namespace NJson;

    TJsonValue jsonConfig;
    TFileInput jsonInput(path);
    Y_ENSURE(ReadJsonTree(&jsonInput, &jsonConfig));
    return FromJson(jsonConfig);
}

TConfig TConfig::FromJson(const NJson::TJsonValue& params) {
    TConfig config;
    config.DumpDirectory = params["dump_directory"].GetString();
    config.Port = static_cast<ui16>(params["port"].GetInteger());
    config.Threads = static_cast<ui16>(params["threads"].GetInteger());
    config.CurrencyRateSource = params["currency_rate_source"].GetString();
    config.StationSegmentsIndex = params["station_segments_index"].GetBoolean();

    for (const auto& item : params["t_types"].GetArray()) {
        config.TransportTypes.insert(Types.at(item.GetString()));
    }
    TVector<TString> languages;
    for (const auto& item : params["languages"].GetArray()) {
        languages.push_back(item.GetString());
    }

    for (const auto& [tld, json_host] : params["desktop_host_by_tld"].GetMap()) {
        config.HostConfig.AddHost(false, json_host.GetString(), tld);
    }
    for (const auto& [tld, json_host] : params["mobile_host_by_tld"].GetMap()) {
        config.HostConfig.AddHost(true, json_host.GetString(), tld);
    }

    config.LanguageConfig = TLanguageConfig{params["default_lang"].GetString(), languages};

    const auto flightStatusesParams = params["flight_statuses"];

    auto allowedStationIds = THashSet<object_id_t>{};
    for (const auto& allowedStationIdValue : flightStatusesParams["allowed_station_ids"].GetArray()) {
        allowedStationIds.insert(static_cast<object_id_t>(allowedStationIdValue.GetInteger()));
    }

    config.FlightStatusesConfig = TFlightStatusesConfig{
        static_cast<ui16>(flightStatusesParams["query_limit"].GetInteger()),
        static_cast<ui16>(flightStatusesParams["timeout_milliseconds"].GetInteger()),
        flightStatusesParams["source"].GetString(),
        allowedStationIds,
    };

    return config;
}

TLanguageConfig::TLanguageConfig(const TString& defaultLanguage, const TVector<TString>& languages) {
    DefaultLanguage = LanguageByName(defaultLanguage);
    for (const auto& lang : languages) {
        Languages[lang] = LanguageByName(lang);
    }
}

ELanguage TLanguageConfig::Get(const TString& lang) const {
    auto it = Languages.find(lang);
    return it == Languages.end() ? DefaultLanguage : it->second;
}

ELanguage TLanguageConfig::GetDefault() const {
    return DefaultLanguage;
}

const THashMap<TString, std::function<TString(TString, ELanguage)>>& TLanguageConfig::GetTranslations() {
    static THashMap<TString, std::function<TString(TString, ELanguage)>> translations = {
        {"(кольцевой)",
         [](TString&& title, ELanguage language) -> TString {
             switch (language) {
                 case LANG_UKR:
                     return title + " (кільцевий)";
                 case LANG_TUR:
                     return title + " (çevre yolu)";
                 default:
                     return title + " (кольцевой)";
             }
         }},
        {"(кольцо)",
         [](TString&& title, ELanguage language) -> TString {
             switch (language) {
                 case LANG_UKR:
                     return title + " (кільце)";
                 case LANG_TUR:
                     return title + " (çevre yolu)";
                 default:
                     return title + " (кольцо)";
             }
         }},
        {"/ кольцевой",
         [](TString&& title, ELanguage language) -> TString {
             switch (language) {
                 case LANG_UKR:
                     return title + "/ кільцевий";
                 case LANG_TUR:
                     return title + "/ çevre hattı";
                 default:
                     return title + "/ кольцевой";
             }
         }},
        {"(согласованные поезда)",
         [](TString&& title, ELanguage language) -> TString {
             switch (language) {
                 case LANG_UKR:
                     return title + " (узгоджені поїзди)";
                 case LANG_TUR:
                     return title + " (devamlı trenler)";
                 default:
                     return title + " (согласованные поезда)";
             }
         }},
        {"(с пересадкой)",
         [](TString&& title, ELanguage language) -> TString {
             return title + " (с пересадкой)";
         }},
        {"(с пересадками)",
         [](TString&& title, ELanguage language) -> TString {
             return title + " (с пересадками)";
         }}};

    return translations;
}

void THostConfig::AddHost(bool isMobile, const TString& host, const TString& tld) {
    THashMap<TString, TString>& hostByTld = (isMobile ? MobileHostByTld : DesktopHostByTld);
    hostByTld[tld] = host;
}

const TString& THostConfig::GetHostByTld(bool isMobile, const TString& tld) const {
    const THashMap<TString, TString>& hostByTld = (isMobile ? MobileHostByTld : DesktopHostByTld);

    auto host = hostByTld.find(tld);
    if (host == hostByTld.end()) {
        return hostByTld.at("ru");
    }

    return host->second;
}
