#include "datetime_helpers.h"

#include <util/string/split.h>

std::pair<NDatetime::TSimpleTM, NDatetime::TSimpleTM> TScheduleRangeProvider::GetRange(const NDatetime::TTimeZone& timezone,
                                                                                       const TMaybe<NDatetime::TSimpleTM>& searchDate,
                                                                                       const int delta) {
    NDatetime::TSimpleTM pointTime = NDatetime::ToCivilTime(ToAbsoluteTime(GetNow(), NDatetime::GetUtcTimeZone()), timezone);
    NDatetime::TSimpleTM pointDate = NDatetime::CreateCivilTime(timezone, pointTime.RealYear(), pointTime.RealMonth(), pointTime.MDay);

    if (!searchDate.Empty()) {
        auto minDate = *searchDate;
        auto maxDate = *searchDate;
        maxDate.Add(NDatetime::TSimpleTM::EField::F_DAY, 1);

        if (searchDate == pointDate) {
            // если запрашивается расписание на сегодня, то фильтруем от
            // текущего времени до конца суток
            return {pointTime, maxDate};
        } else if (searchDate > pointDate) {
            return {minDate, maxDate};
        }
    }

    auto minDate = pointTime;
    auto maxDate = pointTime;
    maxDate.Add(NDatetime::TSimpleTM::EField::F_DAY, delta);

    return {minDate, maxDate};
};

NDatetime::TSimpleTM TScheduleRangeProvider::GetNow() {
    return NDatetime::TSimpleTM::CurrentUTC();
}

TMaybe<NDatetime::TSimpleTM> ParseCivilTime(const NDatetime::TTimeZone& timezone, const TString& rawSearchDate) {
    if (rawSearchDate.empty()) {
        return Nothing();
    }

    ui32 year, month, day;
    try {
        Split(rawSearchDate, '-', year, month, day);
    } catch (yexception) {
        return Nothing();
    }

    return NDatetime::CreateCivilTime(timezone, year, month, day);
};

NDatetime::TSimpleTM AddMinutes(NDatetime::TSimpleTM naiveDateTime, i32 minutes) {
    naiveDateTime.Add(NDatetime::TSimpleTM::F_MIN, minutes);
    return naiveDateTime;
};

NDatetime::TSimpleTM ChangeTimeZone(const NDatetime::TSimpleTM& dateTime, const NDatetime::TTimeZone& originalTimeZone, const NDatetime::TTimeZone& otherTimeZone) {
    return NDatetime::ToCivilTime(NDatetime::ToAbsoluteTime(dateTime, originalTimeZone), otherTimeZone);
}
