#include "flight_status.h"
#include "segment_finders.h"

using namespace NRasp;

TVector<TString> TFlightStatusFetcher::MakeFlightKeys(const TArrayRef<const TSegment>& segments) const {
    auto result = TVector<TString>{};
    for (const auto& segment : segments) {
        const auto& thread = segment.ThreadWrapper();
        const auto flightDate = segment.LocalDepartureTime().ToString("%Y%m%d");
        const auto flightNumber = SubstGlobalCopy(thread.Number(), " ", "");
        result.emplace_back(flightDate + flightNumber);
    }
    return result;
}

TMaybe<const NJson::TJsonValue> TFlightStatusFetcher::FetchFlightStatusesData(
    const TVector<TString>& flightKeys) const {
    const auto postData = "flight=" + JoinSeq(",", flightKeys);
    const auto request = NHttp::TFetchQuery{
        SourceUrl,
        NHttp::TFetchOptions{}.SetPostData(postData).SetTimeout(TDuration::MilliSeconds(TimeoutMilliseconds))};

    const auto startTime = TInstant::Now();
    auto response = NHttp::Fetch(request);
    Cerr << TThread::CurrentThreadId() << " Fetch of " << postData << " done in " << (TInstant::Now() - startTime).SecondsFloat() << " seconds" << Endl;

    if (!response->Success()) {
        Cerr << "response->Code = " << response->Code << Endl;
        return Nothing();
    }

    NJson::TJsonValue flightsJson;
    if (!NJson::ReadJsonTree(response->Data, &flightsJson)) {
        Cerr << "Cannot parse JSON. response->Data.Size() = " << response->Data.Size() << Endl;
        return Nothing();
    }

    return flightsJson;
}

TInstant TFlightStatusFetcher::ParseInstant(const TString& dateTimeValue) const {
    auto parsedTM = tm{};
    time_t time = 0;
    if (strptime(dateTimeValue.Data(), "%Y-%m-%d %H:%M:%S", &parsedTM) == nullptr) {
        Cerr << "Cannot parse tm: " << dateTimeValue << Endl;
    } else {
        time = TimeGM(&parsedTM);
    }
    return TInstant::Seconds(time);
}

NDatetime::TSimpleTM TFlightStatusFetcher::ParseTM(const TString& dateTimeValue) const {
    auto parsedTM = tm{};
    if (strptime(dateTimeValue.Data(), "%Y-%m-%d %H:%M:%S", &parsedTM) == nullptr) {
        Cerr << "Cannot parse tm: " << dateTimeValue << Endl;
        return NDatetime::TSimpleTM();
    }
    return NDatetime::TSimpleTM::New(parsedTM);
}

TMaybe<const TFlightStatus> TFlightStatusFetcher::ParseFlightStatus(
    const NJson::TJsonValue& flightJson,
    const TCommonSegmentFinder::EEventType eventType,
    const NDatetime::TTimeZone& airportTimezone) const {
    const auto& flightStatusJson = flightJson["status"];

    const auto& updatedAtString = flightStatusJson["updated_at"].GetString();
    if (updatedAtString.Empty()) {
        return Nothing();
    }

    const auto& eventString =
        eventType == TCommonSegmentFinder::EEventType::Departure
            ? flightStatusJson["departure"].GetString()
            : flightStatusJson["arrival"].GetString();
    if (eventString.Empty()) {
        return Nothing();
    }

    const auto updateTM = ParseTM(updatedAtString);
    const auto eventInstant = NDatetime::ToAbsoluteTime(ParseTM(eventString), airportTimezone);

    if (eventType == TCommonSegmentFinder::EEventType::Departure) {
        return TFlightStatus{
            eventInstant,
            ParseInstant(flightJson["departure_utc"].GetString()),
            static_cast<object_id_t>(flightJson["airport_from_id"].GetInteger()),
            updateTM,
            flightStatusJson["departure_terminal"].GetString(),
            flightStatusJson["departure_gate"].GetString(),
            flightStatusJson["status"].GetString(),
            "",
            flightStatusJson["check_in_desks"].GetString()};
    } else {
        return TFlightStatus{
            eventInstant,
            ParseInstant(flightJson["arrival_utc"].GetString()),
            static_cast<object_id_t>(flightJson["airport_to_id"].GetInteger()),
            updateTM,
            flightStatusJson["arrival_terminal"].GetString(),
            flightStatusJson["arrival_gate"].GetString(),
            flightStatusJson["status"].GetString(),
            flightStatusJson["baggage_carousels"].GetString(),
            ""};
    }
}

THashMap<TString, const TFlightStatus> TFlightStatusFetcher::ParseFlightStatuses(
    const NJson::TJsonValue& flightsJson,
    const TCommonSegmentFinder::EEventType eventType,
    const NDatetime::TTimeZone& airportTimezone) const {
    auto result = THashMap<TString, const TFlightStatus>{};

    for (const auto& flightJson : flightsJson.GetArray()) {
        const auto flightStatus = ParseFlightStatus(flightJson, eventType, airportTimezone);
        if (flightStatus.Empty()) {
            continue;
        }

        const auto flightKey =
            SubstGlobalCopy(flightJson["departure_day"].GetString(), "-", "") +
            SubstGlobalCopy(flightJson["number"].GetString(), " ", "");

        result.emplace(flightKey, flightStatus.GetRef());
    }

    return result;
}

TFlights TFlightStatusFetcher::Fetch(
    const TStationWrapper& airport,
    const TArrayRef<const TSegment> flightSegments,
    const TCommonSegmentFinder::EEventType eventType) const {
    auto result = TFlights{};

    const auto flightKeys = MakeFlightKeys(flightSegments);
    const auto flightStatusesData = FetchFlightStatusesData(flightKeys);
    const auto flightStatuses = ParseFlightStatuses(
        flightStatusesData.GetOrElse(NJson::TJsonValue{}), eventType, airport.Timezone());

    for (size_t i = 0; i < flightKeys.size(); ++i) {
        const auto& flightKey = flightKeys[i];
        const auto& flightSegment = flightSegments[i];

        const auto flightStatusIt = flightStatuses.find(flightKey);
        if (flightStatusIt == flightStatuses.end()) {
            result.emplace_back(&flightSegment, Nothing());
            continue;
        }

        const auto& flightStatus = flightStatusIt->second;
        const auto& eventInstant =
            eventType == TCommonSegmentFinder::EEventType::Departure
                ? flightSegment.DepartureDt()
                : flightSegment.ArrivalDt();
        if (flightStatus.EventStationId != airport.OuterId() || flightStatus.ScheduledEventInstant != eventInstant) {
            result.emplace_back(&flightSegment, Nothing());
            continue;
        }

        result.emplace_back(&flightSegment, flightStatus);
    }

    return result;
}
