#pragma once

#include "segment.h"
#include "segment_finders.h"

#include <library/cpp/http/client/client.h>
#include <library/cpp/json/json_reader.h>

#include <util/draft/datetime.h>
#include <util/generic/array_ref.h>
#include <util/generic/vector.h>
#include <util/string/join.h>
#include <util/string/subst.h>
#include <util/system/thread.h>

#include <utility>

namespace NRasp {
    struct TFlightStatus {
        TInstant EventInstant;
        TInstant ScheduledEventInstant;
        object_id_t EventStationId;
        NDatetime::TSimpleTM UpdateTM;
        TString Terminal;
        TString Gate;
        TString Status;
        TString BaggageCarousels;
        TString CheckInDesks;
    };

    struct TFlight {
        const TSegment* Segment;
        TMaybe<TFlightStatus> Status;

        TFlight(const TSegment* segment, TMaybe<TFlightStatus> status)
            : Segment(segment)
            , Status(std::move(status)){};

        static const TInstant& GetDepartureInstant(const TFlight& flight) {
            return flight.Status.Defined() ? flight.Status->EventInstant : flight.Segment->DepartureDt();
        }

        static const TInstant& GetArrivalInstant(const TFlight& flight) {
            return flight.Status.Defined() ? flight.Status->EventInstant : flight.Segment->ArrivalDt();
        }
    };

    using TFlightEventInstantGetter = std::function<const TInstant&(const TFlight& flight)>;

    using TFlights = TVector<TFlight>;

    class TFlightStatusFetcher {
    public:
        TFlightStatusFetcher(const TString& sourceUrl, const ui16 timeoutMilliseconds)
            : SourceUrl{sourceUrl}
            , TimeoutMilliseconds{timeoutMilliseconds} {};

        TFlights Fetch(
            const TStationWrapper& airport,
            TArrayRef<const TSegment> flightSegments,
            TCommonSegmentFinder::EEventType eventType) const;

    private:
        TVector<TString> MakeFlightKeys(const TArrayRef<const TSegment>& segments) const;

        TMaybe<const NJson::TJsonValue> FetchFlightStatusesData(const TVector<TString>& flightKeys) const;

        TInstant ParseInstant(const TString& dateTimeValue) const;

        NDatetime::TSimpleTM ParseTM(const TString& dateTimeValue) const;

        TMaybe<const TFlightStatus> ParseFlightStatus(
            const NJson::TJsonValue& flightJson,
            TCommonSegmentFinder::EEventType eventType,
            const NDatetime::TTimeZone& airportTimezone) const;

        THashMap<TString, const TFlightStatus> ParseFlightStatuses(
            const NJson::TJsonValue& flightsJson,
            TCommonSegmentFinder::EEventType eventType,
            const NDatetime::TTimeZone& airportTimezone) const;

    private:
        TString SourceUrl;
        ui16 TimeoutMilliseconds;
    };
}
