#include "limit_condition.h"

#include <util/generic/algorithm.h>
#include <util/generic/vector.h>

using namespace NRasp;

TLimitConditions::TLimitConditions(const TPointKey& fromPointKey, const TPointKey& toPointKey,
                                   const TTransportSet& transportTypes,
                                   const TVector<TStationWrapper>& stations,
                                   bool extended)
    : FromPointKey_(fromPointKey)
    , ToPointKey_(toPointKey)
{
    TStation::EMajority maxMajorityId = TStation::STATION_ID;
    auto fromMaxMajorityID = maxMajorityId;
    auto toMaxMajorityID = maxMajorityId;

    bool isDifferentSettlements = false;
    bool isStationsInTheSameCity = false;
    bool isFromCityToStationInOtherCity = false;
    bool isFromCityToStationInTheSameCity = false;
    bool isFromStationInOtherCityToCity = false;
    bool isFromStationInTheSameCityToCity = false;

    if (fromPointKey.IsSettlement() && toPointKey.IsSettlement()) {
        isDifferentSettlements = true;
    } else if (fromPointKey.IsStation() && toPointKey.IsStation()) {
        const TStationWrapper& fromStation = stations.at(fromPointKey.Id() - 1);
        const TStationWrapper& toStation = stations.at(toPointKey.Id() - 1);

        isStationsInTheSameCity = fromStation.HasSettlement() && toStation.HasSettlement() && fromStation.SettlementId() == toStation.SettlementId();
    } else if (fromPointKey.IsStation()) {
        const auto& fromStation = stations.at(fromPointKey.Id() - 1);
        if (fromStation.HasSettlement() && fromStation.SettlementId() == toPointKey.Id())
            isFromStationInTheSameCityToCity = true;
        else
            isFromStationInOtherCityToCity = true;
    } else {
        const auto& toStation = stations.at(toPointKey.Id() - 1);

        if (toStation.HasSettlement() && fromPointKey.Id() == toStation.SettlementId())
            isFromCityToStationInTheSameCity = true;
        else
            isFromCityToStationInOtherCity = true;
    }

    TTransportSet newTransportTypes;

    if (isDifferentSettlements) {
        fromMaxMajorityID = toMaxMajorityID = TStation::NOT_IN_TABLO_ID;
    } else if (isStationsInTheSameCity || isFromCityToStationInTheSameCity || isFromStationInTheSameCityToCity) {
        if (!extended) {
            for (const auto& type : transportTypes)
                if (!EqualToOneOf(type, TRThread::TRAIN, TRThread::PLANE))
                    newTransportTypes.insert(type);
        }
    } else if (isFromCityToStationInOtherCity) {
        fromMaxMajorityID = TStation::NOT_IN_TABLO_ID;
    } else if (isFromStationInOtherCityToCity) {
        toMaxMajorityID = TStation::NOT_IN_TABLO_ID;
    }

    FromMaxMajorityID_ = fromMaxMajorityID;
    ToMaxMajorityID_ = toMaxMajorityID;
    TransportTypes_ = newTransportTypes.empty() ? transportTypes : newTransportTypes;
}
