#include <utility>

#pragma clang diagnostic ignored "-Wunused-parameter"

#include "airport_panel.h"
#include "point_key.h"
#include "raw_segment.h"
#include "rasp_database.h"
#include "run_time_logger.h"
#include "query_reader.h"
#include "config.h"
#include "rasp_search_index.h"
#include "segment_finders.h"
#include "station_segments_index.h"
#include "tariffs/static_tariff.h"
#include "tariffs/static_tariff_search_index.h"
#include "tariffs/currencies.h"

#include <travel/hotels/lib/cpp/ordinal_date/ordinal_date.h> // todo(mangin): move ordinal_date => to travel/lib
#include <travel/rasp/route-search-api/proto/rthread.pb.h>
#include <travel/rasp/route-search-api/url_builders.h>
#include <travel/rasp/route-search-api/lib/views/proto/direction_response.pb.h>
#include <travel/rasp/route-search-api/lib/views/proto/error_response.pb.h>
#include <travel/rasp/route-search-api/lib/views/proto/event_status.pb.h>
#include <travel/rasp/route-search-api/lib/views/proto/station_direction.pb.h>
#include <travel/rasp/route-search-api/lib/views/proto/station_response.pb.h>

#include <library/cpp/getopt/last_getopt.h>
#include <library/cpp/json/json_value.h>
#include <library/cpp/json/json_reader.h>
#include <library/cpp/http/server/http.h>
#include <library/cpp/http/server/response.h>
#include <library/cpp/http/misc/parsed_request.h>
#include <library/cpp/containers/fast_trie/fasttrie.h>
#include <library/cpp/json/writer/json.h>
#include <library/cpp/json/json_reader.h>

#include <google/protobuf/util/json_util.h>

#include <util/datetime/base.h>
#include <util/folder/path.h>
#include <util/generic/set.h>
#include <util/stream/file.h>
#include <util/datetime/parser.h>
#include <util/system/yassert.h>

using namespace NJson;


class THttpRasp: public THttpServer::ICallBack {
private:
    class THttpRaspRequest: public TClientRequest {
        Y_FORCE_INLINE bool Response(const NJsonWriter::TBuf& json, const HttpCodes& code = HTTP_OK) {
            const auto& body = json.Str();
            Output() << "HTTP/1.1 " << HttpCodeStrEx(code) << "\r\n"
                                                              "Content-Type: application/json; charset=utf-8\r\n"
                                                              "Content-Length: "
                     << body.Size() << "\r\n"
                                       "\r\n"
                     << body;
            return true;
        };

        Y_FORCE_INLINE bool ResponseAsJson(const google::protobuf::Message& message, const HttpCodes& code = HTTP_OK) {
            TString json_string;
            google::protobuf::util::JsonPrintOptions options;

            NProtoBuf::util::MessageToJsonString(message, &json_string);
            Output() << "HTTP/1.1 " << HttpCodeStrEx(code) << "\r\n"
                                                              "Content-Type: application/json; charset=utf-8\r\n"
                                                              "Content-Length: "
                     << json_string.Size() << "\r\n"
                                              "\r\n"
                     << json_string;
            return true;
        };

        Y_FORCE_INLINE bool ResponseAsProtobuf(const google::protobuf::Message& message, const HttpCodes& code = HTTP_OK) {
            TString protobuf_string;
            bool serialized = message.SerializeToString(&protobuf_string);
            Y_ASSERT(serialized);

            Output() << "HTTP/1.1 " << HttpCodeStrEx(code) << "\r\n"
                                                              "Content-Type: application/x-route-search-api; charset=utf-8\r\n"
                                                              "Content-Length: "
                     << protobuf_string.Size() << "\r\n"
                                                  "\r\n"
                     << protobuf_string;
            return true;
        };

        Y_FORCE_INLINE bool Response(const TString& body, const HttpCodes& code = HTTP_OK) {
            Output() << "HTTP/1.1 " << HttpCodeStrEx(code) << "\r\n"
                                                              "Content-Type: text/plain; charset=utf-8\r\n"
                                                              "Content-Length: "
                     << body.Size() << "\r\n"
                                       "\r\n"
                     << body;
            return true;
        };

        Y_FORCE_INLINE bool Error(const HttpCodes& code, const TString& body) {
            NJsonWriter::TBuf json;
            json.BeginObject()
                .WriteKey("error")
                .WriteString(body)
                .WriteKey("code")
                .WriteInt(code)
                .EndObject();
            return Response(json, code);
        }

    public:
        THttpRaspRequest(THttpRasp* server)
            : Server(server)
        {
        }

        bool Ping(const TParsedHttpFull& req) {
            return Response("OK");
        }

        //todo(ashibaev) : если fromPointKey == toPointKey, то сейчас отдаст все расписание из этой точки. Пока оставлю это здесь
        bool FindDirectionSegments(const TParsedHttpFull& req) {
            RunTimeLogger findDirectionSegmentsLogger("FindDirectionSegments");
            TCgiParameters params(req.Cgi);
            NRasp::TDirectionQuery query;

            try {
                query = NRasp::TDirectionQueryReader::ReadHTTPQuery(
                    params, Server->Config.LanguageConfig, Server->Database);
            } catch (const NRasp::TInvalidQueryException& exception) {
                return Error(HTTP_BAD_REQUEST, exception.what());
            }

            RunTimeLogger findDirectionSegmentsbySearcherLogger("Searcher.FindDirectionSegments");
            auto segments = Searcher().FindDirectionSegments(query);
            findDirectionSegmentsbySearcherLogger.Emit();

            if (segments.empty()) {
                if (query.ResponseFormat == NRasp::EResponseFormat::Json) {
                    NRaspProto::NErrorResponse::TErrorResponse response;
                    response.set_error("no routes found");
                    return ResponseAsJson(response);
                }

                return Response("", HTTP_NO_CONTENT);
            }

            NRaspProto::NDirectionResponse::TDirectionResponse pbResponse;

            pbResponse.SetTotal(segments.ysize());

            {
                RunTimeLogger findDirectionSegmentsFormatLogger("FindDirectionSegments format response");

                for (ui32 i = 0; i < Min<ui32>(segments.ysize(), query.Limit); ++i) {
                    auto& segment = segments[i];
                    auto pbSegment = pbResponse.AddSegments();
                    NRasp::TStaticTariffQuery tariffQuery{
                        segment->DepartureWrapper().Station().Id(),
                        segment->ArrivalWrapper().Station().Id(),
                        segment->ThreadWrapper().Id(),
                        segment->DepartureTimeInDay(),
                        query.Language,
                        segment->LocalDepartureTime().RealMonth(),
                        segment->LocalDepartureTime().MDay};

                    RunTimeLogger getStaticTariffLogger("TariffSearcher.GetStaticTariff");
                    auto price = Server->TariffSearcher.GetStaticTariff(tariffQuery);
                    getStaticTariffLogger.Emit();

                    if (!price.Empty()) {
                        pbSegment->MutablePrice()->SetValue(price->BaseValue());
                        pbSegment->MutablePrice()->SetCurrencyId(static_cast<size_t>(price->Currency()));
                    }

                    auto departureTimestamp = pbSegment->MutableDepartureTimestamp();
                    departureTimestamp->SetTimestamp(static_cast<ui32>(segment->DepartureDt().TimeT()));
                    departureTimestamp->SetUtcOffset(segment->LocalDepartureTime().GMTOff);

                    pbSegment->SetDepartureStationId(segment->DepartureWrapper().Station().OuterId());

                    auto arrivalTimestamp = pbSegment->MutableArrivalTimestamp();
                    arrivalTimestamp->SetTimestamp(static_cast<ui32>(segment->ArrivalDt().TimeT()));
                    arrivalTimestamp->SetUtcOffset(segment->LocalArrivalTime().GMTOff);

                    pbSegment->SetArrivalStationId(segment->ArrivalWrapper().Station().OuterId());

                    auto thread = pbSegment->MutableThread();
                    thread->SetStartDate(727007); // это поле устаревшее, в потребителях не используется
                    thread->SetNumber(segment->ThreadWrapper().Number());
                    thread->SetTitle(segment->ThreadWrapper().Title(query.Language, Server->Database));

                    auto urls = pbSegment->MutableUrls();
                    urls->SetMobile(NRasp::NUrlBuilders::BuildThreadUrl(true, *segment, query, Server->Config.HostConfig));
                    urls->SetDesktop(NRasp::NUrlBuilders::BuildThreadUrl(false, *segment, query, Server->Config.HostConfig));
                }
            }

            if (query.ResponseFormat == NRasp::EResponseFormat::Json) {
                return ResponseAsJson(pbResponse);
            }

            return ResponseAsProtobuf(pbResponse);
        }

        void DumpAirportFlights(
            const NRasp::TFlights& flights,
            NRaspProto::NStationResponse::TStationDirection* pbDirection,
            const NRasp::TStationQuery& query) {
            pbDirection->SetTotal(flights.size());

            for (ui32 i = 0; i < Min<ui32>(flights.size(), query.Limit); ++i) {
                const auto& flight = flights[i];
                const auto& segment = *flight.Segment;
                const auto pbSegment = pbDirection->AddSegments();

                auto thread = pbSegment->MutableThread();
                thread->SetNumber(segment.ThreadWrapper().Number());
                thread->SetTitle(segment.ThreadWrapper().Title(query.Language, Server->Database));
                thread->SetCompanyId(segment.ThreadWrapper().CompanyId());

                auto urls = pbSegment->MutableUrls();
                urls->SetMobile(NRasp::NUrlBuilders::BuildThreadUrl(true, segment, query, Server->Config.HostConfig));
                urls->SetDesktop(NRasp::NUrlBuilders::BuildThreadUrl(false, segment, query, Server->Config.HostConfig));

                auto departureTimestamp = pbSegment->MutableDepartureTimestamp();
                departureTimestamp->SetTimestamp(static_cast<ui32>(segment.DepartureDt().TimeT()));
                departureTimestamp->SetUtcOffset(segment.LocalDepartureTime().GMTOff);

                pbSegment->SetDepartureStationId(segment.DepartureWrapper().Station().OuterId());
                pbSegment->SetDepartureTerminalId(segment.DepartureWrapper().TerminalId());

                auto arrivalTimestamp = pbSegment->MutableArrivalTimestamp();
                arrivalTimestamp->SetTimestamp(static_cast<ui32>(segment.ArrivalDt().TimeT()));
                arrivalTimestamp->SetUtcOffset(segment.LocalArrivalTime().GMTOff);

                pbSegment->SetArrivalStationId(segment.ArrivalWrapper().Station().OuterId());
                pbSegment->SetArrivalTerminalId(segment.ArrivalWrapper().TerminalId());

                if (flight.Status.Defined()) {
                    const auto& flightStatus = flight.Status.GetRef();
                    auto pbEventStatus = pbSegment->MutableEventStatus();

                    pbEventStatus->SetTimestamp(flightStatus.EventInstant.TimeT());
                    pbEventStatus->SetTerminal(flightStatus.Terminal);
                    pbEventStatus->SetGate(flightStatus.Gate);

                    auto statusValue = flightStatus.Status;
                    statusValue.to_upper();
                    NRaspProto::NSegment::TEventStatus::EStatus status;
                    pbEventStatus->EStatus_Parse(statusValue, &status);
                    pbEventStatus->SetStatus(status);
                    pbEventStatus->SetBaggageCarousels(flightStatus.BaggageCarousels);
                    pbEventStatus->SetCheckInDesks(flightStatus.CheckInDesks);
                }
            }
        }

        bool FindAirportFlights(const TParsedHttpFull& req) {
            RunTimeLogger findAirportFlightsLogger("FindAirportFlights");

            const auto& airportPanelPtr = Server->AirportPanel;
            if (airportPanelPtr.Empty()) {
                return Error(HTTP_SERVICE_UNAVAILABLE, "Station segments index is disabled");
            }

            TCgiParameters params(req.Cgi);

            if (!params.Has("station_id")) {
                return Error(HTTP_BAD_REQUEST, "Parameter `station_id` is required");
            }

            const auto stationId = params.Get("station_id");
            const auto eventDate = params.Has("event_date") ? params.Get("event_date") : TString("");

            const auto tld = params.Get("tld");
            const auto limit = params.Has("limit") ? FromString<ui32>(params.Get("limit")) : 3;
            const auto& language = Server->Config.LanguageConfig.Get(params.Has("lang") ? params.Get("lang") : "");

            const auto format = params.Has("format") ? params.Get("format") : "protobuf";
            const auto isJson = format == "json";
            const auto isProtobuf = format == "protobuf";
            if (!isJson && !isProtobuf) {
                return Error(HTTP_BAD_REQUEST, "Incorrect format of response");
            }

            auto query = NRasp::TStationQueryReader::MakeQuery(stationId, eventDate, limit, language, tld, Server->Database);
            if (query.Empty()) {
                return Error(HTTP_BAD_REQUEST, "Incorrect format of query");
            }

            RunTimeLogger findFlightsLogger("TariffSearcher.GetStaticTariff");
            const auto [departureSegments, arrivalSegments] = airportPanelPtr->FindFlights(query.GetRef());
            findFlightsLogger.Emit();

            if (!departureSegments.size() && !arrivalSegments.size()) {
                if (isProtobuf) {
                    return Response("", HTTP_NO_CONTENT);
                } else if (isJson) {
                    NRaspProto::NErrorResponse::TErrorResponse response;
                    response.set_error("no routes found");
                    return ResponseAsJson(response);
                }
            }

            NRaspProto::NStationResponse::TStationResponse pbResponse;

            const auto pbDepartureDirection = pbResponse.AddDirections();
            pbDepartureDirection->SetType(NRaspProto::NStationResponse::TStationDirection::DEPARTURE);
            {

                RunTimeLogger dumpAirportFlightsLogger("DumpAirportFlights departureSegments");
                DumpAirportFlights(departureSegments, pbDepartureDirection, query.GetRef());
            }

            const auto pbArrivalDirection = pbResponse.AddDirections();
            pbArrivalDirection->SetType(NRaspProto::NStationResponse::TStationDirection::ARRIVAL);
            {
                RunTimeLogger dumpAirportFlightsLogger("DumpAirportFlights arrivalSegments");
                DumpAirportFlights(arrivalSegments, pbArrivalDirection, query.GetRef());
            }

            if (isProtobuf) {
                return ResponseAsProtobuf(pbResponse);
            }
            return ResponseAsJson(pbResponse);
        }

        bool Reply(void*) {
            const auto& firstLine = Input().FirstLine();
            try {
                TParsedHttpFull req(firstLine);
                TString path(req.Path);

                if (!Server->router.Find(path)) {
                    return Error(HTTP_NOT_FOUND, "Not Found");
                }

                const auto& route = Server->router[path];

                return (this->*route)(req);
            } catch (yexception) {
                Cerr << firstLine << " (400): " << CurrentExceptionMessage() << Endl;
                return Error(HTTP_BAD_REQUEST, CurrentExceptionMessage());
            } catch (...) {
                Cerr << firstLine << " (500): " << CurrentExceptionMessage() << Endl;
                return Error(HTTP_INTERNAL_SERVER_ERROR, CurrentExceptionMessage());
            }
        }

    private:
        const NRasp::TSegmentSearcherWithContext& Searcher() const {
            return Server->Searcher;
        }

        const NRasp::TRaspSearchIndex& SearchIndex() const {
            return Server->SearchIndex;
        }

        const TMaybe<const NRasp::TAirportPanel>& AirportPanel() const {
            return Server->AirportPanel;
        }

        THttpRasp* Server;
    };

public:
    using THandler = bool (THttpRaspRequest::*)(const TParsedHttpFull&);
    TFastTrie<THandler> router;

    THttpRasp(const NRasp::TWrappedRaspDatabase& database, const NRasp::TRaspSearchIndex& searchIndex,
              const NRasp::TSegmentSearcher& searcher, const NRasp::TStaticTariffSearchIndex& tariffSearchIndex,
              const TMaybe<NRasp::TStationSegmentsIndex>& stationSegmentsIndex, NRasp::TConfig config)
        : Database(database)
        , Searcher(searcher, searchIndex)
        , TariffSearcher(tariffSearchIndex)
        , SearchIndex(searchIndex)
        , AirportPanel(MakeAirportPanel(searchIndex, stationSegmentsIndex, config.FlightStatusesConfig))
        , Config(std::move(config))
    {
        router.Insert(TString("/api/bus-direction/"), &THttpRaspRequest::FindDirectionSegments); // TODO убрать после обновления прокси
        router.Insert(TString("/api/direction/"), &THttpRaspRequest::FindDirectionSegments);
        router.Insert(TString("/api/station/"), &THttpRaspRequest::FindAirportFlights);
        router.Insert(TString("/ping"), &THttpRaspRequest::Ping);
    }

    TClientRequest* CreateClient() override {
        return new THttpRaspRequest(this);
    }

private:
    const NRasp::TWrappedRaspDatabase& Database;
    NRasp::TSegmentSearcherWithContext Searcher;
    NRasp::TStaticTariffServiceWithContext TariffSearcher;
    const NRasp::TRaspSearchIndex& SearchIndex;
    TMaybe<const NRasp::TAirportPanel> AirportPanel;
    NRasp::TConfig Config;

    static TMaybe<const NRasp::TAirportPanel> MakeAirportPanel(
        const NRasp::TRaspSearchIndex& searchIndex,
        const TMaybe<NRasp::TStationSegmentsIndex>& stationSegmentsIndex,
        const NRasp::TFlightStatusesConfig& flightStatusConfig) {
        if (stationSegmentsIndex.Defined()) {
            return {
                NRasp::TAirportPanel{searchIndex, stationSegmentsIndex.GetRef(), flightStatusConfig}};
        }
        return {};
    }
};

bool ParseArgs(int argc, char** argv, TString& filename) {
    using namespace NLastGetopt;
    TOpts opts;
    TOpt& configFilename = opts
                               .AddLongOption("config", "json config")
                               .Required()
                               .RequiredArgument();

    THolder<TOptsParseResult> r;
    r.Reset(new TOptsParseResult(&opts, argc, argv));

    filename = r->Get(&configFilename);
    if (!NFs::Exists(filename)) {
        Cerr << filename << " doesn't exist!" << Endl;
        return false;
    }
    return true;
}

// todo(ashibaev)
/*
 * Очевидно, что внутренние id - это неудобно.
 * Надо подумать над тем, как избежать их полностью или сделать их удобными
 * 1. Как вариант - явные называния (Id -> InnerId, StationId -> InnerStationId)
 *    В этом случае ошибиться до сих пор просто, и это ничем не проверяется, кроме как глазами
 * 2. Создать 2 класса, которые являются обертками над size_t для того, чтобы такие вещи
 *    проверялись на моменте компиляции.
 * 3. Полностью отказаться от внутренних Id и использовать HashMap вместо вектора.
 *    Надо посмотреть, что будет с памятью и временем.
 */

int main(int argc, char** argv) {
    using namespace NRasp;

    TString filename;

    if (!ParseArgs(argc, argv, filename)) {
        return 1;
    }

    auto config = TConfig::FromFile(filename);
    LoadRates(config.CurrencyRateSource);

    Clog << "Reading data... ";
    RunTimeLogger readingDataLogger("Reading data");

    TRaspDatabaseReader databaseReader;
    TRaspDatabase database;

    try {
        database = databaseReader.Read(config.DumpDirectory);
    } catch (TRaspDatabaseReaderException& e) {
        Cerr << e.what() << Endl;
        return 1;
    }
    for (const auto& currency : database.GetItems<TCurrencyProto>()) {
        TCurrency::AddCurrency(currency.code(), currency.id());
    }

    readingDataLogger.Emit();

    Clog << "Stations : " << database.GetItems<TStation>().size() << Endl;
    Clog << "Settlements : " << database.GetItems<TSettlement>().size() << Endl;
    Clog << "Threads : " << database.GetItems<TRThread>().size() << Endl;
    Clog << "RTStations : " << database.GetItems<TThreadStation>().size() << Endl;

    Clog << "Initializing database... ";
    RunTimeLogger initDatabaseLogger("Initializing database");
    TWrappedRaspDatabase wrappedRaspDatabase{database};
    initDatabaseLogger.Emit();

    Clog << "Building search index... ";
    RunTimeLogger buildingSearchIndexLogger("Building search index");
    TRaspSearchIndex searchIndex{wrappedRaspDatabase, config.TransportTypes};
    TSegmentSearcher searcher{};
    buildingSearchIndexLogger.Emit();

    Clog << "Building tariffs index... ";
    RunTimeLogger buildingTariffsIndexLogger("Building tariffs index");
    TStaticTariffSearchIndex tariffSearchIndex{wrappedRaspDatabase, config.TransportTypes};
    buildingTariffsIndexLogger.Emit();

    TMaybe<TStationSegmentsIndex> stationSegmentsIndex;
    if (config.StationSegmentsIndex) {
        Clog << "Building station segments index... ";
        RunTimeLogger buildingStationSegmentsIndexLogger("Building station segments index");
        stationSegmentsIndex = {TStationSegmentsIndex{wrappedRaspDatabase, searchIndex, config.TransportTypes}};
    }

    try {
        THttpRasp httpRasp{wrappedRaspDatabase, searchIndex, searcher, tariffSearchIndex, stationSegmentsIndex, config};

        THttpServer::TOptions options;
        options.Port = config.Port;
        options.nThreads = config.Threads;
        options.MaxConnections = 1024;
        options.MaxQueueSize = 4096;
        options.KeepAliveEnabled = true;
        options.CompressionEnabled = false;

        THttpServer server{&httpRasp, options};
        server.Start();
        Clog << "Started." << Endl;

        server.Wait();
        Clog << "Stopped." << Endl;

        return 0;
    } catch (...) {
        Cerr << CurrentExceptionMessage() << Endl;
        return 1;
    }
}
