#pragma once

#include "helpers.h"
#include "wrappers.h"

#include <travel/rasp/route-search-api/proto/rtstation.pb.h>

#include <util/generic/hash.h>
#include <util/generic/vector.h>
#include <utility>

namespace NRasp {
    // Обертка над HashMap, которая хранит список ключей.
    template <typename TKey, typename TValue>
    class THashMapKeysWrapper {
    public:
        using iterator = typename THashMap<TKey, TValue>::iterator;
        using value_type = std::pair<TKey, TValue>;

        THashMapKeysWrapper() {
        }

        THashMapKeysWrapper(const std::initializer_list<value_type>& list)
            : InnerMap(list)
        {
            AddKeys();
        }

        THashMapKeysWrapper(const THashMap<TKey, TValue>& innerMap)
            : InnerMap(innerMap)
        {
            AddKeys();
        }

        THashMapKeysWrapper(const THashMapKeysWrapper& keysWrapper) = default;

        TValue& operator[](const TKey& key);

        TValue& at(const TKey& key);
        const TValue& at(const TKey& key) const;

        size_t size() const {
            return InnerMap.size();
        }

        std::pair<iterator, bool> insert(const value_type& obj);

        // Сортирует ключи
        // Можно было хранить Set ключей и поддерживать этот инвариант.
        // Не стал так делать, так как сортировка нам понадобится лишь один раз,
        // после чего мы уже часто будем пробегаться по ключам.
        void Normalize();

        const TVector<TKey>& GetKeys() const;

        inline auto begin() {
            return InnerMap.begin();
        }

        inline auto end() {
            return InnerMap.end();
        }

        inline auto begin() const {
            return InnerMap.begin();
        }

        inline auto end() const {
            return InnerMap.end();
        }

        void clear() {
            InnerMap.clear();
        }

        // Находит общие ключи с другим THashMapKeysWrapper
        // Никак не проверяет отсортированность ключей
        TVector<TKey> GetCommonKeysWith(const THashMapKeysWrapper& other) const;

    private:
        void AddKeys() {
            for (auto& p : InnerMap) {
                Keys.push_back(p.first);
            }
        }

        THashMap<TKey, TValue> InnerMap;
        TVector<TKey> Keys;
    };

    template <typename TKey, typename TValue>
    void THashMapKeysWrapper<TKey, TValue>::Normalize() {
        SortUnique(Keys);
    }

    template <typename TKey, typename TValue>
    TValue& THashMapKeysWrapper<TKey, TValue>::operator[](const TKey& key) {
        auto it = InnerMap.find(key);
        if (it == InnerMap.end()) {
            Keys.push_back(key);
            return InnerMap[key];
        }
        return it->second;
    }

    template <typename TKey, typename TValue>
    const TVector<TKey>& THashMapKeysWrapper<TKey, TValue>::GetKeys() const {
        return Keys;
    }

    template <typename TKey, typename TValue>
    TValue& THashMapKeysWrapper<TKey, TValue>::at(const TKey& key) {
        auto it = InnerMap.find(key);
        if (it == InnerMap.end()) {
            Keys.push_back(key);
            return InnerMap[key];
        }
        return it->second;
    }

    template <typename TKey, typename TValue>
    const TValue& THashMapKeysWrapper<TKey, TValue>::at(const TKey& key) const {
        return InnerMap.at(key);
    }

    template <typename TKey, typename TValue>
    TVector<TKey> THashMapKeysWrapper<TKey, TValue>::GetCommonKeysWith(const THashMapKeysWrapper& other) const {
        TVector<TKey> commonThreads;

        SetIntersection(Keys.begin(), Keys.end(),
                        other.Keys.begin(), other.Keys.end(),
                        std::back_inserter(commonThreads));

        return commonThreads;
    }

    template <typename TKey, typename TValue>
    std::pair<typename THashMapKeysWrapper<TKey, TValue>::iterator, bool>
    THashMapKeysWrapper<TKey, TValue>::insert(const THashMapKeysWrapper::value_type& obj) {
        if (!InnerMap.contains(obj.first)) {
            Keys.push_back(obj.first);
        }
        return InnerMap.insert(obj);
    }
}
