#pragma once

#include "helpers.h"

#include <travel/rasp/route-search-api/proto/settlement.pb.h>
#include <travel/rasp/route-search-api/proto/station.pb.h>

#include <util/generic/string.h>

namespace NRasp {
    class TPointKey {
    public:
        enum class EPointKeyType {
            Settlement,
            Station
        };

        static constexpr char settlementPrefix = 'c';
        static constexpr char stationPrefix = 's';

        TPointKey() {
        }

        TPointKey(object_id_t id, EPointKeyType type)
            : Type_(type)
            , Id_(id)
        {
        }

        TPointKey(const TPointKey& pointKey) = default;

        inline bool operator==(const TPointKey& rhs) const noexcept;

        inline bool operator!=(const TPointKey& rhs) const noexcept;

        bool TryParseFromString(const TString& stringPointKey);
        TString ToString() const;

        inline EPointKeyType Type() const noexcept;

        inline bool IsSettlement() const noexcept;

        inline bool IsStation() const noexcept;

        inline object_id_t Id() const noexcept;

        template <typename T>
        inline TPointKey ToInnerIds(const T& database) const {
            if (IsStation()) {
                return TPointKey(database.template GetInnerId<TStation>(Id_), Type_);
            }
            return TPointKey(database.template GetInnerId<TSettlement>(Id_), Type_);
        }

    private:
        EPointKeyType Type_;
        object_id_t Id_;
    };

    inline bool TPointKey::operator==(const TPointKey& rhs) const noexcept {
        return Id_ == rhs.Id_ && Type_ == rhs.Type_;
    }

    inline bool TPointKey::operator!=(const TPointKey& rhs) const noexcept {
        return Id_ != rhs.Id_ || Type_ != rhs.Type_;
    }

    inline TPointKey::EPointKeyType TPointKey::Type() const noexcept {
        return Type_;
    }

    inline bool TPointKey::IsSettlement() const noexcept {
        return Type_ == EPointKeyType::Settlement;
    }

    inline bool TPointKey::IsStation() const noexcept {
        return Type_ == EPointKeyType::Station;
    }

    inline object_id_t TPointKey::Id() const noexcept {
        return Id_;
    }
}
