#pragma once

#include "travel/rasp/route-search-api/config.h"
#include "travel/rasp/route-search-api/datetime_helpers.h"
#include "travel/rasp/route-search-api/point_key.h"
#include "travel/rasp/route-search-api/proto/settlement.pb.h"
#include "travel/rasp/route-search-api/rasp_database.h"

#include <library/cpp/timezone_conversion/convert.h>

#include <util/datetime/parser.h>
#include <util/generic/maybe.h>
#include <util/stream/input.h>
#include <library/cpp/cgiparam/cgiparam.h>
#include <util/string/split.h>

namespace NRasp {
    class TInvalidQueryException: public yexception {
    };

    enum class EResponseFormat {
        Protobuf,
        Json
    };

    class TDirectionQuery {
    public:
        TPointKey FromPointKey;
        TPointKey ToPointKey;
        TMaybe<NDatetime::TSimpleTM> SearchDate;
        NDatetime::TSimpleTM MinDepartureTime;
        NDatetime::TSimpleTM MaxDepartureTime;
        TString Tld;
        TString MainReqid;
        TString ThreadNumber;
        ui32 Limit;
        ELanguage Language;
        EResponseFormat ResponseFormat;

        TDirectionQuery() = default;

        TDirectionQuery(const TPointKey& fromPointKey, const TPointKey& toPointKey,
                        const TMaybe<NDatetime::TSimpleTM>& searchDate,
                        const NDatetime::TSimpleTM minDeparutreTime,
                        const NDatetime::TSimpleTM maxDepartureTime,
                        const TString& tld,
                        const TString& main_reqid,
                        const TString threadNumber = "",
                        const ui32 limit = 5,
                        const ELanguage language = ELanguage::LANG_RUS,
                        const EResponseFormat responseFormat = EResponseFormat::Protobuf)
            : FromPointKey(fromPointKey)
            , ToPointKey(toPointKey)
            , SearchDate(searchDate)
            , MinDepartureTime(minDeparutreTime)
            , MaxDepartureTime(maxDepartureTime)
            , Tld(tld)
            , MainReqid(main_reqid)
            , ThreadNumber(threadNumber)
            , Limit(limit)
            , Language(language)
            , ResponseFormat(responseFormat)
        {
        }

        template <typename T>
        TDirectionQuery ToInnerIds(const T& info) const {
            auto result = TDirectionQuery(
                FromPointKey.ToInnerIds(info),
                ToPointKey.ToInnerIds(info),
                SearchDate,
                MinDepartureTime,
                MaxDepartureTime,
                Tld,
                MainReqid,
                ThreadNumber,
                Limit,
                Language,
                ResponseFormat);
            return result;
        }
    };

    class TStationQuery {
    public:
        object_id_t StationId;
        TMaybe<NDatetime::TSimpleTM> EventDate;
        NDatetime::TSimpleTM MinEventTM;
        NDatetime::TSimpleTM MaxEventTM;
        ui32 Limit;
        ELanguage Language;
        TString Tld;

        TStationQuery(const object_id_t& stationId,
                      const TMaybe<NDatetime::TSimpleTM>& eventDate,
                      const NDatetime::TSimpleTM minEventTM,
                      const NDatetime::TSimpleTM maxEventTM,
                      const ui32 limit,
                      const ELanguage language,
                      const TString& tld)
            : StationId(stationId)
            , EventDate(eventDate)
            , MinEventTM(minEventTM)
            , MaxEventTM(maxEventTM)
            , Limit(limit)
            , Language(language)
            , Tld(tld)
        {
        }

        template <typename T>
        TStationQuery ToInnerIds(const T& info) const {
            return TStationQuery(
                info.template GetInnerId<TStation>(StationId),
                EventDate,
                MinEventTM,
                MaxEventTM,
                Limit,
                Language,
                Tld);
        }
    };

    class IQueryReader {
    public:
        virtual TMaybe<TDirectionQuery> ReadQuery(IInputStream&, const NRasp::TWrappedRaspDatabase& database) const = 0;
    };

    class TDirectionQueryReader : IQueryReader {
        // Парсит запросы вида: c213 s10111 2018-08-10

    public:
        static TPointKey ParsePointKey(const TString pointKeyValue, const NRasp::TWrappedRaspDatabase& database) {
            TPointKey pointKey;

            if (!pointKey.TryParseFromString(pointKeyValue)) {
                throw TInvalidQueryException() << "Invalid point key value " << pointKeyValue;
            }

            if (!database.ExistsByPointKey(pointKey)) {
                throw TInvalidQueryException() << "Unknown point key value " << pointKeyValue;
            }

            return pointKey;
        }

        static TDirectionQuery MakeQuery(
            const NRasp::TWrappedRaspDatabase& database,
            const TString& departurePointKeyValue,
            const TString& arrivalPointKeyValue,
            const TString& dateString,
            const TString& threadNumber,
            const ui32 limit,
            const ELanguage language,
            const TString& tld,
            const TString& main_reqid,
            const EResponseFormat responseFormat) {
            const auto departurePointKey = ParsePointKey(departurePointKeyValue, database);
            const auto arrivalPointKey = ParsePointKey(arrivalPointKeyValue, database);
            const auto& timezone = database.GetTimezoneIn(departurePointKey);
            auto searchDate = ParseCivilTime(timezone, dateString);
            auto [minDate, maxDate] = TScheduleRangeProvider().GetRange(timezone, searchDate);

            return {
                departurePointKey,
                arrivalPointKey,
                searchDate,
                minDate,
                maxDate,
                tld,
                main_reqid,
                threadNumber,
                limit,
                language,
                responseFormat};
        }

        static EResponseFormat ParseFormat(const TString& formatValue) {
            if (formatValue.Empty() || formatValue == "protobuf") {
                return EResponseFormat::Protobuf;
            }

            if (formatValue == "json") {
                return EResponseFormat::Json;
            }

            throw TInvalidQueryException()
                << "Invalid format of response " << formatValue;
        }

        static TDirectionQuery ReadHTTPQuery(
            const TCgiParameters& params,
            const NRasp::TLanguageConfig& languageConfig,
            const NRasp::TWrappedRaspDatabase& database) {
            const auto& departurePointKeyValue = params.Get("departure_point_key");
            const auto& arrivalPointKeyValue = params.Get("arrival_point_key");

            if (departurePointKeyValue.Empty() || arrivalPointKeyValue.Empty()) {
                throw TInvalidQueryException()
                    << "Parameters `departure_point_key` and `arrival_point_key` are required";
            }

            const auto& date = params.Get("departure_date");
            const auto& tld = params.Get("tld");
            const auto& main_reqid = params.Get("main_reqid");
            const auto& threadNumber = params.Get("thread_number");
            const auto limit = params.Has("limit") ? FromString<ui32>(params.Get("limit")) : 5;
            const auto language = languageConfig.Get(params.Get("lang"));
            const auto format = ParseFormat(params.Get("format"));

            return MakeQuery(
                database,
                departurePointKeyValue,
                arrivalPointKeyValue,
                date,
                threadNumber,
                limit,
                language,
                tld,
                main_reqid,
                format);
        }

        TMaybe<TDirectionQuery> ReadQuery(IInputStream& stream, const NRasp::TWrappedRaspDatabase& database) const override {
            TString departurePointKeyValue, arrivalPointKeyValue, dateString, tld, main_reqid;
            TString line;

            stream.ReadLine(line);
            if (line.Empty()) {
                return Nothing();
            }

            Split(line, " ", departurePointKeyValue, arrivalPointKeyValue, dateString, tld, main_reqid);

            return MakeQuery(
                database,
                departurePointKeyValue,
                arrivalPointKeyValue,
                dateString,
                "",
                5,
                ELanguage::LANG_RUS,
                tld,
                main_reqid,
                EResponseFormat::Protobuf);
        }
    };

    class TStationQueryReader {
    public:
        static TMaybe<TStationQuery> MakeQuery(const TString& stationIdString, const TString& eventDateString,
                                               const ui32 limit,
                                               const ELanguage& language, const TString& tld,
                                               const NRasp::TWrappedRaspDatabase& database) {
            TMaybe<TStationQuery> query;
            object_id_t stationId;

            if (!TryIntFromString<10>(stationIdString, stationId)) {
                return query;
            }

            if (!database.Exists<TStation>(stationId)) {
                return query;
            }

            const auto& timezone = database.GetStationTimezone(stationId);
            auto eventDate = ParseCivilTime(timezone, eventDateString);
            auto [minEventDate, maxEventDate] = TScheduleRangeProvider().GetRange(timezone, eventDate, 3);

            query.ConstructInPlace(stationId, eventDate, minEventDate, maxEventDate, limit, language, tld);

            return query;
        }
    };
}
