#include "rasp_database.h"

#include <travel/rasp/route-search-api/proto/rasp_precache.pb.h>

using namespace NRasp;

bool NRasp::CheckDirectory(const TString& directory) {
    static TVector<TString> names = {"rthread", "rtstation", "settlement", "station",
                                     "station2settlement", "threadtariff", "supplier", "precache", "currency"};

    for (const auto& name : names) {
        if (!NFs::Exists(JoinFsPaths(directory, name))) {
            return false;
        }
    }
    return NFs::Exists(JoinFsPaths(directory, "precache", "precache"));
}

TString NRasp::GetFilename(int currentFileId, const TString& directory) {
    TString res = ToString(currentFileId);
    while (res.size() < 5)
        res = "0" + res;
    return JoinFsPaths(directory, res);
}

template <typename TPack,
          typename TObject = typename std::remove_reference<decltype(*(TPack().mutable_items()->begin()))>::type>
TVector<TObject> TRaspDatabaseReader::ReadFromDirectory(const TString& directory, const TMaybe<std::function<bool(const TObject&)>> predicatePtr) const {
    TVector<TObject> result;
    int fileId = 0;
    TString filename = GetFilename(fileId, directory);
    auto reader = TObjectStreamReader<TPack, TObject>();

    for (; NFs::Exists(filename); fileId++, filename = GetFilename(fileId, directory)) {
        TFileInput input(filename);
        auto objects = reader.Read(input);

        if (predicatePtr.Defined()) {
            const auto predicate = predicatePtr.GetRef();
            for (const auto& object : objects) {
                if (predicate(object)) {
                    result.push_back(object);
                }
            }
        } else {
            result.insert(result.end(), objects.begin(), objects.end());
        }
    }
    return result;
}

TRaspDatabase TRaspDatabaseReader::Read(TString dumpDirectory) {
    if (!CheckDirectory(dumpDirectory)) {
        ythrow TRaspDatabaseReaderException() << "Can't parse directory " << dumpDirectory
                                              << " in directory " << NFs::CurrentWorkingDirectory();
    }

    auto getDumpDirectory = [&](const TString& name) { return JoinFsPaths(dumpDirectory, name); };

    auto settlements = ReadFromDirectory<TSettlementPack>(getDumpDirectory("settlement"), {IsEnabled<TSettlement>});
    auto stations = ReadFromDirectory<TStationPack>(getDumpDirectory("station"), {IsEnabled<TStation>});
    auto rtstations = ReadFromDirectory<TThreadStationPack>(getDumpDirectory("rtstation"));
    auto rthreads = ReadFromDirectory<TRThreadPack>(getDumpDirectory("rthread"), {IsEnabled<TRThread>});
    auto stationToSettlements = ReadFromDirectory<TStation2SettlementPack>(getDumpDirectory("station2settlement"));
    auto threadTariffs = ReadFromDirectory<TThreadTariffPack>(getDumpDirectory("threadtariff"));
    auto supplier = ReadFromDirectory<TSupplierPack>(getDumpDirectory("supplier"));
    auto idToTimezones = ReadTimezones(getDumpDirectory("precache"));
    auto uids = ReadUids(getDumpDirectory("precache"));
    auto currencies = ReadFromDirectory<TCurrencyPack>(getDumpDirectory("currency"));

    return TRaspDatabase(std::move(idToTimezones),
                         std::move(rthreads),
                         std::move(rtstations),
                         std::move(settlements),
                         std::move(stations),
                         std::move(stationToSettlements),
                         std::move(threadTariffs),
                         std::move(supplier),
                         std::move(uids),
                         std::move(currencies));
}

THashMap<int, NDatetime::TTimeZone> TRaspDatabaseReader::ReadTimezones(const TString& directory) const {
    TFileInput input(JoinFsPaths(directory, "precache"));
    auto reader = TTimezonesStreamReader();
    return reader.Read(input);
}

THashMap<object_id_t, TString> TRaspDatabaseReader::ReadUids(const TString& directory) const {
    TFileInput input(JoinFsPaths(directory, "precache"));
    auto reader = TUidCacheStreamReader();
    return reader.Read(input);
}

template <class TWrapper>
void TWrappedRaspDatabase::BuildCache(TVector<TWrapper>& wrappers) {
    const auto& objects = Database.GetItems<typename TWrapper::TBase>();
    for (const auto& object : objects)
        wrappers.push_back(TWrapper(object, *this));
}

TWrappedRaspDatabase::TWrappedRaspDatabase(const TRaspDatabase& database)
    : Database(database)
{
    InitNormalizer(RThreadNormalizer);
    InitNormalizer(RTStationNormalizer);
    InitNormalizer(SettlementNormalizer);
    InitNormalizer(StationNormalizer);
    InitNormalizer(ThreadTariffNormalizer);
    InitNormalizer(SupplierNormalizer);

    // Тут важен порядок!
    BuildCache(SettlementCache);
    BuildCache(SupplierCache);
    BuildCache(StationCache);
    BuildCache(ThreadCache);
    BuildCache(RTStationCache);
    BuildCache(ThreadTariffCache);
    BuildCache(StationToSettlementsCache);
}

TRaspDatabase::TRaspDatabase(const THashMap<int, NDatetime::TTimeZone>& idToTimezone,
                             const TVector<TRThread>& rthreads,
                             const TVector<TThreadStation>& rtstations,
                             const TVector<TSettlement>& settlements,
                             const TVector<TStation>& stations,
                             const TVector<TStation2Settlement>& stationToSettlements,
                             const TVector<TThreadTariff>& threadTariffs,
                             const TVector<TSupplier>& suppliers,
                             const THashMap<object_id_t, TString>& uidById,
                             const TVector<TCurrencyProto>& currencies)
    : IdToTimezone(idToTimezone)
    , RThreads(rthreads)
    , RTStations(rtstations)
    , Settlements(settlements)
    , Stations(stations)
    , StationToSettlements(stationToSettlements)
    , ThreadTariffs(threadTariffs)
    , Suppliers(suppliers)
    , UidById(uidById)
    , Currencies(currencies)
{
}

TRaspDatabase::TRaspDatabase(THashMap<int, NDatetime::TTimeZone>&& idToTimezone,
                             TVector<TRThread>&& rthreads,
                             TVector<TThreadStation>&& rtstations,
                             TVector<TSettlement>&& settlements,
                             TVector<TStation>&& stations,
                             TVector<TStation2Settlement>&& stationToSettlements,
                             TVector<TThreadTariff>&& threadTariffs,
                             TVector<TSupplier>&& suppliers,
                             THashMap<object_id_t, TString>&& uidById,
                             TVector<TCurrencyProto>&& currencies)
    : IdToTimezone(idToTimezone)
    , RThreads(rthreads)
    , RTStations(rtstations)
    , Settlements(settlements)
    , Stations(stations)
    , StationToSettlements(stationToSettlements)
    , ThreadTariffs(threadTariffs)
    , Suppliers(suppliers)
    , UidById(uidById)
    , Currencies(currencies)
{
}

THashMap<int, NDatetime::TTimeZone> TTimezonesStreamReader::Read(IInputStream& input) const {
    THashMap<int, NDatetime::TTimeZone> idToTimezone;
    TRaspPrecache precache;
    precache.ParseFromArcadiaStream(&input);
    for (const auto& p : precache.timezone_ids()) {
        try {
            idToTimezone[p.second] = NDatetime::GetTimeZone(p.first);
        } catch (NDatetime::TInvalidTimezone&) {
            idToTimezone[p.second] = NDatetime::GetTimeZone("Europe/Moscow");
        }
    }
    return idToTimezone;
}

THashMap<object_id_t, TString> TUidCacheStreamReader::Read(IInputStream& input) const {
    THashMap<object_id_t, TString> uidById;
    TRaspPrecache precache;
    precache.ParseFromArcadiaStream(&input);
    for (const auto& p : precache.uid_to_id()) {
        uidById[p.second] = p.first;
    }

    return uidById;
}
