#pragma once

#include "id_normalizer.h"
#include "wrappers.h"
#include "point_key.h"

#include <travel/rasp/route-search-api/proto/rthread.pb.h>
#include <travel/rasp/route-search-api/proto/rtstation.pb.h>
#include <travel/rasp/route-search-api/proto/settlement.pb.h>
#include <travel/rasp/route-search-api/proto/station.pb.h>
#include <travel/rasp/route-search-api/proto/station_to_settlement.pb.h>
#include <travel/rasp/route-search-api/proto/tariff.pb.h>
#include <travel/rasp/route-search-api/proto/supplier.pb.h>

#include <library/cpp/timezone_conversion/civil.h>

#include <util/folder/path.h>
#include <util/generic/hash.h>
#include <util/generic/vector.h>
#include <util/generic/string.h>
#include <util/stream/file.h>
#include <util/system/fs.h>
#include <util/string/builder.h>

namespace NRasp {
    static const TString empty = "";

    class TRaspDatabaseReaderException: public yexception {
    };

    // Хранит информацию о всех Station, Settlement, RThread, RTStation, StationToSettlements, кроме скрытых
    class TRaspDatabase {
    private:
        THashMap<int, NDatetime::TTimeZone> IdToTimezone;
        TVector<TRThread> RThreads;
        TVector<TThreadStation> RTStations;
        TVector<TSettlement> Settlements;
        TVector<TStation> Stations;
        TVector<TStation2Settlement> StationToSettlements;
        TVector<TThreadTariff> ThreadTariffs;
        TVector<TSupplier> Suppliers;
        THashMap<object_id_t, TString> UidById;
        TVector<TCurrencyProto> Currencies;

    public:
        TRaspDatabase() = default;

        TRaspDatabase(const THashMap<int, NDatetime::TTimeZone>& idToTimezone,
                      const TVector<TRThread>& rthreads,
                      const TVector<TThreadStation>& rtstations,
                      const TVector<TSettlement>& settlements,
                      const TVector<TStation>& stations,
                      const TVector<TStation2Settlement>& stationToSettlements,
                      const TVector<TThreadTariff>& threadTariffs = {},
                      const TVector<TSupplier>& suppliers = {},
                      const THashMap<object_id_t, TString>& uidById = {},
                      const TVector<TCurrencyProto>& currencies = {});

        TRaspDatabase(THashMap<int, NDatetime::TTimeZone>&& idToTimezone,
                      TVector<TRThread>&& rthreads,
                      TVector<TThreadStation>&& rtstations,
                      TVector<TSettlement>&& settlements,
                      TVector<TStation>&& stations,
                      TVector<TStation2Settlement>&& stationToSettlements,
                      TVector<TThreadTariff>&& threadTariffs = {},
                      TVector<TSupplier>&& suppliers = {},
                      THashMap<object_id_t, TString>&& uidById = {},
                      TVector<TCurrencyProto>&& currencies = {});

        // Информация о соответствии поля time_zone часовому поясу.
        // В отличие от модели, вместо строки в поле time_zone у proto-объектов хранится число -
        // номер соответсвующей таймзоны в отсортированном списке всех таймзон
        inline const NDatetime::TTimeZone& GetTimezoneById(int id) const {
            return IdToTimezone.at(id);
        }

        // Получит список всех объектов типа T
        template <typename T>
        inline const TVector<T>& GetItems() const noexcept {
            return GetItems(TQueryType<T>());
        }

        const TString& GetUid(object_id_t id) const {
            return UidById.contains(id) ? UidById.at(id) : empty;
        }

    private:
        inline const TVector<TRThread>& GetItems(TQueryType<TRThread>) const {
            return RThreads;
        }

        inline const TVector<TSettlement>& GetItems(TQueryType<TSettlement>) const {
            return Settlements;
        }

        inline const TVector<TStation>& GetItems(TQueryType<TStation>) const {
            return Stations;
        }

        inline const TVector<TThreadStation>& GetItems(TQueryType<TThreadStation>) const {
            return RTStations;
        }

        inline const TVector<TStation2Settlement>& GetItems(TQueryType<TStation2Settlement>) const {
            return StationToSettlements;
        }

        inline const TVector<TThreadTariff>& GetItems(TQueryType<TThreadTariff>) const {
            return ThreadTariffs;
        }

        inline const TVector<TSupplier>& GetItems(TQueryType<TSupplier>) const {
            return Suppliers;
        }

        inline const TVector<TCurrencyProto>& GetItems(TQueryType<TCurrencyProto>) const {
            return Currencies;
        }
    };

    // Хранит обертки над объектами TRaspDatabase
    // Обертки отличаются тем, что имеют нормализованные id
    class TWrappedRaspDatabase {
    private:
        TIdNormalizer<TRThread> RThreadNormalizer;
        TIdNormalizer<TThreadStation> RTStationNormalizer;
        TIdNormalizer<TSettlement> SettlementNormalizer;
        TIdNormalizer<TStation> StationNormalizer;
        TIdNormalizer<TThreadTariff> ThreadTariffNormalizer;
        TIdNormalizer<TSupplier> SupplierNormalizer;

        TVector<TRThreadWrapper> ThreadCache;
        TVector<TStationWrapper> StationCache;
        TVector<TSettlementWrapper> SettlementCache;
        TVector<TThreadStationWrapper> RTStationCache;
        TVector<TStationToSettlementWrapper> StationToSettlementsCache;
        TVector<TThreadTariffWrapper> ThreadTariffCache;
        TVector<TSupplierWrapper> SupplierCache;

        const TRaspDatabase& Database;

    public:
        explicit TWrappedRaspDatabase(const TRaspDatabase& database);

        template <typename T>
        inline const TVector<T>& GetItems() const noexcept {
            return GetItems(TQueryType<T>());
        }

        inline const NDatetime::TTimeZone& GetTimezoneById(int id) const {
            return Database.GetTimezoneById(id);
        }

        inline NDatetime::TTimeZone GetStationTimezone(const object_id_t& stationId) const {
            return GetItemWithId<TStationWrapper>(GetInnerId<TStation>(stationId)).Timezone();
        }

        inline NDatetime::TTimeZone GetSettlementTimezone(const object_id_t& settlementId) const {
            return GetItemWithId<TSettlementWrapper>(GetInnerId<TSettlement>(settlementId)).Timezone();
        }

        inline NDatetime::TTimeZone GetTimezoneIn(const TPointKey& pointKey) const {
            return pointKey.IsStation() ? GetStationTimezone(pointKey.Id()) : GetSettlementTimezone(pointKey.Id());
        }

        inline bool ExistsByPointKey(const TPointKey& pointKey) const {
            return pointKey.IsStation() ? Exists<TStation>(pointKey.Id()) : Exists<TSettlement>(pointKey.Id());
        }

        template <typename T>
        bool Exists(object_id_t id) const {
            return GetInnerId<T>(id) != 0;
        }

        template <typename T>
        inline object_id_t GetInnerId(object_id_t id) const {
            return GetInnerId(id, TQueryType<T>());
        }

        template <typename T>
        inline const T& GetItemWithId(object_id_t id) const {
            Y_ASSERT(1u <= id && id <= GetItems<T>().size());
            return GetItems<T>().at(id - 1);
        }

        template <typename T>
        inline const T& GetItemWithOuterId(object_id_t id) const {
            return GetItemWithId<T>(GetInnerId(id, TQueryType<typename T::TBase>()));
        }

        template <typename T>
        inline yssize_t GetMaxId() const {
            return GetItems<T>().ysize();
        }

        inline const TString& GetUid(object_id_t id) const {
            return Database.GetUid(id);
        }

    private:
        template <class TWrapper>
        void BuildCache(TVector<TWrapper>& wrappers);

        template <typename T>
        void InitNormalizer(TIdNormalizer<T>& normalizer) {
            normalizer = TIdNormalizer<T>(Database.GetItems<T>());
        }

        inline const TVector<TSettlementWrapper>& GetItems(TQueryType<TSettlementWrapper>) const {
            return SettlementCache;
        }

        inline const TVector<TStationWrapper>& GetItems(TQueryType<TStationWrapper>) const {
            return StationCache;
        }

        inline const TVector<TThreadStationWrapper>& GetItems(TQueryType<TThreadStationWrapper>) const {
            return RTStationCache;
        }

        inline const TVector<TRThreadWrapper>& GetItems(TQueryType<TRThreadWrapper>) const {
            return ThreadCache;
        }

        inline const TVector<TStationToSettlementWrapper>& GetItems(TQueryType<TStationToSettlementWrapper>) const {
            return StationToSettlementsCache;
        }

        inline const TVector<TThreadTariffWrapper>& GetItems(TQueryType<TThreadTariffWrapper>) const {
            return ThreadTariffCache;
        }

        inline const TVector<TSupplierWrapper>& GetItems(TQueryType<TSupplierWrapper>) const {
            return SupplierCache;
        }

        inline object_id_t GetInnerId(object_id_t id, TQueryType<TRThread>) const {
            return RThreadNormalizer.GetId(id);
        }

        inline object_id_t GetInnerId(object_id_t id, TQueryType<TSettlement>) const {
            return SettlementNormalizer.GetId(id);
        }

        inline object_id_t GetInnerId(object_id_t id, TQueryType<TStation>) const {
            return StationNormalizer.GetId(id);
        }

        inline object_id_t GetInnerId(object_id_t id, TQueryType<TThreadStation>) const {
            return RTStationNormalizer.GetId(id);
        }

        inline object_id_t GetInnerId(object_id_t id, TQueryType<TThreadTariff>) const {
            return ThreadTariffNormalizer.GetId(id);
        }

        inline object_id_t GetInnerId(object_id_t id, TQueryType<TSupplier>) const {
            return SupplierNormalizer.GetId(id);
        }
    };

    template <typename TPack,
              typename TObject = typename std::remove_reference<decltype(*(TPack().mutable_items()->begin()))>::type>
    class TObjectStreamReader {
    public:
        TVector<TObject> Read(IInputStream& input);
    };

    template <typename TPack, typename TObject>
    TVector<TObject> TObjectStreamReader<TPack, TObject>::Read(IInputStream& input) {
        TPack pack;
        pack.ParseFromArcadiaStream(&input);
        TVector<TObject> result;
        result.insert(result.end(), pack.items().begin(), pack.items().end());
        return result;
    }

    class TTimezonesStreamReader {
    public:
        THashMap<int, NDatetime::TTimeZone> Read(IInputStream& input) const;
    };

    class TUidCacheStreamReader {
    public:
        THashMap<object_id_t, TString> Read(IInputStream& input) const;
    };

    class TRaspDatabaseReader {
    public:
        TRaspDatabase Read(TString dumpDirectory);

    private:
        template <typename TPack, typename TObject>
        TVector<TObject> ReadFromDirectory(
            const TString& directory,
            TMaybe<std::function<bool(const TObject&)>> predicate = Nothing()) const;

        THashMap<int, NDatetime::TTimeZone> ReadTimezones(const TString& directory) const;

        THashMap<object_id_t, TString> ReadUids(const TString& directory) const;

        template <typename TObject>
        static bool IsEnabled(const TObject& object) {
            return !object.is_hidden();
        }
    };

    bool CheckDirectory(const TString& directory);

    TString GetFilename(int currentFileId, const TString& directory);
}
