#include "rasp_search_index.h"

using namespace NRasp;

THashMap<object_id_t, TVector<object_id_t>> NRasp::FetchThreadStations(const TVector<TThreadStationWrapper>& rtstations) {
    THashMap<object_id_t, TVector<object_id_t>> threadStations;
    for (const auto& rtstation : rtstations) {
        if (rtstation.HasThread() && rtstation.HasStation()) {
            threadStations[rtstation.ThreadId()].push_back(rtstation.Id());
        }
    }
    return threadStations;
}

THashMap<object_id_t, TVector<object_id_t>> NRasp::FetchStationSettlements(
    const TVector<TStationWrapper>& stations,
    const TVector<TStationToSettlementWrapper>& stationToSettlements) {
    THashMap<object_id_t, TVector<object_id_t>> result;
    for (const auto& station : stations) {
        if (station.HasSettlement())
            result[station.Id()].push_back(station.SettlementId());
    }
    for (const auto& p : stationToSettlements)
        result[p.StationId()].push_back(p.SettlementId());
    for (auto& record : result)
        SortUnique(record.second);
    return result;
}

void NRasp::NormalizePointStops(TPointStops& pointStops) {
#pragma clang diagnostic push
#pragma clang diagnostic ignored "-Wunused-variable"
    pointStops.Normalize();
    for (auto& [threadId, threadStops] : pointStops) {
        SortUniqueBy(threadStops, [](const TThreadStationWrapper* x) { return x->Id(); });
        if (threadStops.empty())
            continue;
    }
#pragma clang diagnostic pop
}

TRaspSearchIndex::TRaspSearchIndex(TWrappedRaspDatabase& database,
                                   const TTransportSet& transportTypes)
    : Database(database)
    , TransportTypes(transportTypes)
    , SettlementStops(database.GetMaxId<TSettlementWrapper>() + 1)
    , StationStops(database.GetMaxId<TStationWrapper>() + 1)
{
    auto stationSettlements = FetchStationSettlements(
        database.GetItems<TStationWrapper>(),
        database.GetItems<TStationToSettlementWrapper>());

    auto threadStations = FetchThreadStations(database.GetItems<TThreadStationWrapper>());

    for (const auto& rthread : database.GetItems<TRThreadWrapper>()) {
        BuildCacheForThread(rthread, threadStations, stationSettlements);
    }
    for (auto& pointStops : SettlementStops) {
        NormalizePointStops(pointStops);
    }
    for (auto& pointStops : StationStops) {
        NormalizePointStops(pointStops);
    }
}

void TRaspSearchIndex::BuildCacheForThread(const TRThreadWrapper& rthread,
                                           const THashMap<object_id_t, TVector<object_id_t>>& threadStations,
                                           const THashMap<object_id_t, TVector<object_id_t>>& stationSettlements) {
    object_id_t threadId = rthread.Id();
    if (rthread.TypeId() == TRThread::CANCEL_ID || !TransportTypes.contains(rthread.TransportType()) || !threadStations.contains(threadId)) {
        return;
    }

    for (auto rtstationId : threadStations.at(threadId)) {
        const auto& rtstation = Database.GetItemWithId<TThreadStationWrapper>(rtstationId);

        if (rtstation.ArrivalOffset().Empty()) {
            ThreadDepartureStop[threadId] = &rtstation;
        } else if (rtstation.DepartureOffset().Empty()) {
            ThreadArrivalStop[threadId] = &rtstation;
        }

        if ((rtstation.DepartureOffset().Defined() && rtstation.ArrivalOffset().Defined() && rtstation.DepartureOffset().GetRef() == rtstation.ArrivalOffset().GetRef()) || rtstation.IsTechnicalStop() || rtstation.StationMajority() >= TStation::NOT_IN_SEARCH_ID) {
            continue;
        }
        object_id_t stationId = rtstation.StationId();
        StationStops[stationId][threadId].push_back(&rtstation);
        if (!stationSettlements.contains(stationId))
            continue;
        for (auto settlementId : stationSettlements.at(stationId)) {
            SettlementStops[settlementId][threadId].push_back(&rtstation);
        }
    }
}

const TMaybe<const TThreadStationWrapper*> TRaspSearchIndex::GetThreadDepartureStop(const TRThreadWrapper& thread) const {
    auto stopIt = ThreadDepartureStop.find(thread.Id());
    if (stopIt == ThreadDepartureStop.end()) {
        return {};
    }
    return {stopIt->second};
}

const TMaybe<const TThreadStationWrapper*> TRaspSearchIndex::GetThreadArrivalStop(const TRThreadWrapper& thread) const {
    auto stopIt = ThreadArrivalStop.find(thread.Id());
    if (stopIt == ThreadArrivalStop.end()) {
        return {};
    }
    return {stopIt->second};
}

const TPointStops& TRaspSearchIndex::GetStationStops(const object_id_t stationId) const {
    return StationStops.at(stationId);
}

const TPointStops& TRaspSearchIndex::GetSettlementStops(const object_id_t settlementId) const {
    return SettlementStops.at(settlementId);
}

const TPointStops& TRaspSearchIndex::GetPointStops(const TPointKey& pointKey) const {
    return pointKey.IsStation() ? GetStationStops(pointKey.Id()) : GetSettlementStops(pointKey.Id());
}
