#pragma once

#include "travel/rasp/route-search-api/limit_condition.h"
#include "travel/rasp/route-search-api/point_key.h"
#include "travel/rasp/route-search-api/rasp_database.h"
#include "travel/rasp/route-search-api/helpers.h"
#include "travel/rasp/route-search-api/map_keys_wrapper.h"

#include <util/generic/map.h>
#include <util/generic/ptr.h>
#include <util/generic/vector.h>
#include <util/generic/hash_set.h>

namespace NRasp {
    using TPointStops = THashMapKeysWrapper<object_id_t, TVector<const TThreadStationWrapper*>>;

    class TRaspSearchIndex {
    private:
        const TWrappedRaspDatabase& Database;
        TTransportSet TransportTypes;

        // Храним для станции по номеру нитки список всех rtstation, лежащих на этой нитке и на этой станции
        // Для городов аналогично
        TVector<TPointStops> SettlementStops;
        TVector<TPointStops> StationStops;

        THashMap<object_id_t, const TThreadStationWrapper*> ThreadDepartureStop;
        THashMap<object_id_t, const TThreadStationWrapper*> ThreadArrivalStop;

    public:
        TRaspSearchIndex(TWrappedRaspDatabase& database, const TTransportSet& transportTypes);

        const TMaybe<const TThreadStationWrapper*> GetThreadDepartureStop(const TRThreadWrapper& thread) const;

        const TMaybe<const TThreadStationWrapper*> GetThreadArrivalStop(const TRThreadWrapper& thread) const;

        const TPointStops& GetPointStops(const TPointKey& pointKey) const;

        const TPointStops& GetStationStops(const object_id_t stationId) const;

        const TPointStops& GetSettlementStops(const object_id_t settlementId) const;

        template <typename T>
        inline const TVector<T>& GetItems() const {
            return Database.GetItems<T>();
        }

        inline const NDatetime::TTimeZone& GetTimezoneById(int id) const {
            return Database.GetTimezoneById(id);
        }

        template <typename T>
        inline object_id_t GetInnerId(object_id_t id) const {
            return Database.GetInnerId<T>(id);
        }

        template <typename T>
        inline const T& GetItemWithId(int id) const {
            return Database.GetItemWithId<T>(id);
        }

        template <typename T>
        inline yssize_t GetMaxId() const {
            return Database.GetMaxId<T>();
        }

        inline const TTransportSet& GetTransportTypes() const {
            return TransportTypes;
        }

    private:
        void BuildCacheForThread(const TRThreadWrapper& thread,
                                 const THashMap<object_id_t, TVector<object_id_t>>& threadStations,
                                 const THashMap<object_id_t, TVector<object_id_t>>& stationSettlements);
    };

    void NormalizePointStops(TPointStops& pointStops);

    // Получаем для каждой станции список номеров settlements, которым она принадлежит
    THashMap<object_id_t, TVector<object_id_t>> FetchStationSettlements(
        const TVector<TStationWrapper>& stations,
        const TVector<TStationToSettlementWrapper>& stationToSettlements);

    // Для каждой нитки список всех rtstation на этой нитке
    THashMap<object_id_t, TVector<object_id_t>> FetchThreadStations(const TVector<TThreadStationWrapper>& rtstations);
}
